(function($){
	'use strict';

	if(typeof pluxixClient === 'undefined'){ return; }
	var i18n = pluxixClient.i18n || {};

	function t(key){ return (i18n && i18n[key]) ? i18n[key] : key; }

	function escHtml(str){
		return String(str)
			.replace(/&/g,'&amp;')
			.replace(/</g,'&lt;')
			.replace(/>/g,'&gt;')
			.replace(/"/g,'&quot;')
			.replace(/'/g,'&#039;');
	}

	function ajax(action, data){
		data = data || {};
		data.action = action;
		data.nonce = pluxixClient.nonce;
		return $.ajax({
			url: pluxixClient.ajaxUrl,
			type: 'POST',
			dataType: 'json',
			data: data
		});
	}

	function notify(type, message){
		var $toast = $('<div class="rsc-toast" />');
		$toast.text(message);
		$toast.addClass(type === 'success' ? 'rsc-toast--success' : 'rsc-toast--error');
		$('body').append($toast);
		setTimeout(function(){ $toast.addClass('rsc-toast--show'); }, 10);
		setTimeout(function(){ $toast.removeClass('rsc-toast--show'); }, 3400);
		setTimeout(function(){ $toast.remove(); }, 3800);
	}

function ajaxError(xhr){
	try{
		if(xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message){
			return xhr.responseJSON.data.message;
		}
		if(xhr && xhr.responseText){
			var txt = String(xhr.responseText);
			// Strip HTML to make it readable inside toast.
			txt = txt.replace(/<[^>]*>/g, ' ');
			txt = txt.replace(/\s+/g, ' ').trim();
			if(txt.length > 220){ txt = txt.substring(0,220) + '...'; }
			if(xhr.status){ return t('msg_error') + ' (' + xhr.status + '): ' + txt; }
			return txt || t('msg_error');
		}
	}catch(e){}
	return t('msg_error');
}

	function openModal(title, bodyHtml, footerHtml){
		closeModal();
		var html = '' +
			'<div class="rsc-modalOverlay" id="rsc-modalOverlay">' +
				'<div class="rsc-modal" role="dialog" aria-modal="true">' +
					'<div class="rsc-modal__header">' +
						'<h3 class="rsc-modal__title">' + escHtml(title) + '</h3>' +
						'<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-modalClose">' + escHtml(t('btn_close')) + '</button>' +
					'</div>' +
					'<div class="rsc-modal__body">' + bodyHtml + '</div>' +
					(footerHtml ? '<div class="rsc-modal__header" style="border-top:1px solid rgba(0,0,0,.08);border-bottom:0;">' + footerHtml + '</div>' : '') +
				'</div>' +
			'</div>';
		$('body').append(html);
		$('#rsc-modalClose').on('click', closeModal);
		$('#rsc-modalOverlay').on('click', function(e){ if(e.target && e.target.id === 'rsc-modalOverlay'){ closeModal(); }});
	}

	function closeModal(){ $('#rsc-modalOverlay').remove(); }

	function shell(title, actionsHtml){
		var $app = $('#rs-client-app');
		var html = '' +
			'<div class="rsc-shell">' +
				'<div class="rsc-topbar">' +
					'<h2 class="rsc-title">' + escHtml(title) + '</h2>' +
					'<div class="rsc-actions">' + (actionsHtml || '') + '</div>' +
				'</div>' +
				'<div class="rsc-content" id="rsc-content"></div>' +
			'</div>';
		$app.html(html);
		return $('#rsc-content');
	}

	function badge(access){
		if(access === 'pro'){
			return '<span class="rsc-badge rsc-badge--pro">' + escHtml(t('badge_pro')) + '</span>';
		}
		return '<span class="rsc-badge rsc-badge--free">' + escHtml(t('badge_free')) + '</span>';
	}

	function accessTypeLabel(access){
		if(!access){ return t('label_access_free'); }
		var type = access.access_type || 'free';
		if(type === 'hosting_partner'){ return t('label_hosting_partner'); }
		if(type === 'manual'){ return t('label_manual'); }
		if(type === 'subscription'){ return t('label_subscription'); }
		return t('label_access_free');
	}

	function renderDashboard(){
		var $c = shell(t('nav_dashboard'),
			'<button type="button" class="rsc-btn" id="rsc-recheck">' + escHtml(t('btn_recheck_access')) + '</button>' +
			'<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-refresh">' + escHtml(t('btn_refresh')) + '</button>'
		);

		$c.html(
			'<div class="rsc-grid" id="rsc-dash-cards"></div>' +
			'<div style="margin-top:14px;" id="rsc-dash-messages"></div>'
		);

		function render(access){
			var cards = [];
			var level = access && access.access_level ? access.access_level : 'free';
			cards.push({label: t('label_access'), value: level === 'pro' ? t('label_access_pro') : t('label_access_free')});
			cards.push({label: t('label_access_type'), value: accessTypeLabel(access)});
			cards.push({label: t('label_domain'), value: pluxixClient.siteDomain || ''});

			if(access && access.subscription && access.subscription.end_date_display){
				cards.push({label: t('label_end_date'), value: access.subscription.end_date_display});
			}

			var html = '';
			cards.forEach(function(c){
				html += '<div class="rsc-card"><div class="rsc-card__label">' + escHtml(c.label) + '</div><div class="rsc-card__value">' + escHtml(c.value) + '</div></div>';
			});
			$('#rsc-dash-cards').html(html);

			var msgs = '';

			if(access && access.ip_based_access){
				msgs += '<div class="rsc-card"><div class="rsc-card__label">' + escHtml(t('label_hosting_partner')) + '</div>' +
					'<div class="rsc-item__desc">' + escHtml(t('msg_hosting_partner')) + '</div>' +
					(access.hosting_label ? '<div class="rsc-item__meta" style="margin-top:8px;">' + escHtml(access.hosting_label) + '</div>' : '') +
				'</div>';
			}

			if(access && access.access_level !== 'pro'){
				if(access.reason === 'domain_not_registered'){
					msgs += '<div class="rsc-lock">' + escHtml(t('msg_domain_required')) + '</div>';
				} else if(access.reason){
					msgs += '<div class="rsc-lock">' + escHtml(t('msg_access_denied')) + '</div>';
				}
			}

			if(access && access.action_url){
				var label = (access.reason === 'domain_not_registered') ? t('cta_register_domain') : t('cta_upgrade');
				msgs += '<div style="margin-top:12px;"><a class="rsc-btn" href="' + escHtml(access.action_url) + '" target="_blank" rel="noopener">' + escHtml(label) + '</a></div>';
			}

			$('#rsc-dash-messages').html(msgs || '<div class="rsc-muted">' + escHtml(t('msg_no_data')) + '</div>');
		}

		function load(){
			$('#rsc-dash-cards').html('<div class="rsc-card"><div class="rsc-card__label">' + escHtml(t('msg_loading')) + '</div></div>');
			$('#rsc-dash-messages').html('<div class="rsc-muted">' + escHtml(t('msg_loading')) + '</div>');
			ajax('rs_client_get_access', {})
				.done(function(res){
					if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
					render(res.data.access || {});
				})
				.fail(function(xhr){ notify('error', ajaxError(xhr)); });
		}

		load();
		$(document).off('click','#rsc-refresh').on('click','#rsc-refresh', load);
		$(document).off('click','#rsc-recheck').on('click','#rsc-recheck', function(){
			ajax('rs_client_sync', {}).done(function(){ notify('success', t('msg_done')); load(); }).fail(function(xhr){ notify('error', ajaxError(xhr)); });
		});

	}

	function renderMarketplace(type){
		var title = (type === 'theme') ? t('nav_themes') : t('nav_plugins');
		var $c = shell(title,
			'<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-refresh">' + escHtml(t('btn_refresh')) + '</button>'
		);

		$c.html(
			'<div id="rsc-market-notice"></div>' +
			'<div class="rsc-toolbar">' +
				'<input type="text" class="rsc-input" id="rsc-search" placeholder="' + escHtml(t('label_search')) + '" />' +
				'<select class="rsc-select" id="rsc-category"><option value="">' + escHtml(t('label_all')) + '</option></select>' +
			'</div>' +
			'<div class="rsc-cards" id="rsc-items"></div>' +
			'<div class="rsc-pager" id="rsc-pager"></div>'
		);

		var access = {};
		var categories = [];
		var itemIndex = {};
		var debounceTimer = null;
		var page = 1;
		var perPage = 12;
		var lastQueryKey = '';
		var loadingMore = false;
		var revalidateInFlight = false;
		var receivedItems = [];

		function renderMarketNotice(){
			if(access && access.access_level === 'pro'){
				$('#rsc-market-notice').empty();
				return;
			}
			var url = (access && access.action_url) ? access.action_url : '#';
			$('#rsc-market-notice').html(
				'<div class="rsc-notice">' +
					'<div class="rsc-notice__body">' +
						'<div class="rsc-notice__title">' + escHtml(t('notice_free_title')) + '</div>' +
						'<div class="rsc-notice__text">' + escHtml(t('notice_free_text')) + '</div>' +
					'</div>' +
					'<div class="rsc-notice__actions">' +
						'<a class="rsc-btn" href="' + escHtml(url) + '" target="_blank" rel="noopener">' + escHtml(t('cta_upgrade')) + '</a>' +
					'</div>' +
				'</div>'
			);
		}

		function lockReason(){
			if(access && access.reason === 'domain_not_registered'){
				return t('msg_domain_required');
			}
			return t('msg_access_denied');
		}

		function renderItems(items, append){
			if(!append){
				itemIndex = {};
				$('#rsc-items').empty();
			}
			if(!items || !items.length){
				if(!append){
					$('#rsc-items').html('<div class="rsc-muted">' + escHtml(t('msg_no_data')) + '</div>');
				}
				return;
			}

			var html = '';
			items.forEach(function(item){
				itemIndex[item.slug] = item;
				var isPro = (item.access_level === 'pro');
				var allowed = !isPro || (access && access.access_level === 'pro');

				var actions = '';
				// Preview is available for themes.
				var canPreview = (type === 'theme') && (item.screenshot_url || item.changelog || item.description);
				if(allowed){
					if(!item.local_installed){
						actions += '<button class="rsc-btn rsc-install" data-slug="' + escHtml(item.slug) + '" data-type="' + escHtml(type) + '">' + escHtml(t('btn_install')) + '</button>';
					}else{
						if(item.update_available){
							actions += '<button class="rsc-btn rsc-update" data-slug="' + escHtml(item.slug) + '" data-type="' + escHtml(type) + '">' + escHtml(t('btn_update')) + '</button>';
						}
						if(type === 'plugin' && !item.local_active){
							actions += '<button class="rsc-btn rsc-btn--ghost rsc-activate" data-slug="' + escHtml(item.slug) + '">' + escHtml(t('btn_activate')) + '</button>';
						}
					}
				} else {
					actions += '<a class="rsc-btn" target="_blank" rel="noopener" href="' + escHtml(access.action_url || '#') + '">' + escHtml((access.reason==='domain_not_registered') ? t('cta_register_domain') : t('cta_upgrade')) + '</a>';
				}


				if(item.demo_url){
					actions += '<a class="rsc-btn rsc-btn--ghost" href="' + escHtml(item.demo_url) + '" target="_blank" rel="noopener">' + escHtml(t('btn_demo')) + '</a>';
				}

				if(canPreview){
					actions += '<button class="rsc-btn rsc-btn--ghost rsc-preview" data-slug="' + escHtml(item.slug) + '">' + escHtml(t('btn_preview')) + '</button>'; 
				}

				var desc = item.description ? item.description : '';
				var meta = '';
				meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_version')) + ': <strong>' + escHtml(item.version || '-') + '</strong></span>';
				if(item.local_installed){
					meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_installed')) + ': <strong>' + escHtml(item.local_version || '-') + '</strong></span>';
					if(type === 'plugin'){
						meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_status')) + ': <strong>' + escHtml(item.local_active ? t('label_active') : t('label_inactive')) + '</strong></span>';
					}
				}else{
					meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_installed')) + ': <strong>' + escHtml(t('label_not_installed')) + '</strong></span>';
				}
				if(item.update_available){
					meta += '<span class="rsc-item__metaPill rsc-item__metaPill--update">' + escHtml(t('btn_update')) + '</span>';
				}

				var thumbUrl = '';
				if(type === 'theme'){
					thumbUrl = item.screenshot_url || item.image_url || '';
				} else {
					thumbUrl = item.image_url || '';
				}
				var thumb = '';
				if(thumbUrl){
					thumb = '<img class="rsc-item__img" src="' + escHtml(thumbUrl) + '" alt="" loading="lazy" />';
				} else {
					thumb = '<div class="rsc-item__imgPlaceholder">' + escHtml((item.name||item.slug||'').substring(0,1).toUpperCase() || 'R') + '</div>';
				}

				html += '' +
					'<div class="rsc-item">' +
						'<div class="rsc-item__media">' + thumb + '</div>' +
						'<div class="rsc-item__body">' +
							'<div class="rsc-item__header">' +
								'<div>' +
									'<div class="rsc-item__titleRow">' +
										'<h3 class="rsc-item__name">' + escHtml(item.name || item.slug) + '</h3>' +
										'<div class="rsc-item__badges">' + badge(item.access_level) + '</div>' +
									'</div>' +
									'<div class="rsc-item__metaRow">' + meta + '</div>' +
								'</div>' +
							'</div>' +
							(desc ? '<div class="rsc-item__desc">' + escHtml(desc) + '</div>' : '') +
							(!allowed ? '<div class="rsc-lock">' + escHtml(lockReason()) + '</div>' : '') +
							'<div class="rsc-item__actions">' + actions + '</div>' +
						'</div>' +
					'</div>';
			});

			if(append){
				$('#rsc-items').append(html);
			} else {
				$('#rsc-items').html(html);
			}
		}

		function renderCategories(cats){
			categories = cats || [];
			var opts = '<option value="">' + escHtml(t('label_all')) + '</option>';
			categories.forEach(function(c){
				opts += '<option value="' + escHtml(c.id) + '">' + escHtml(c.name) + '</option>';
			});
			$('#rsc-category').html(opts);
		}

		function renderPager(p){
			var total = (p && p.total) ? parseInt(p.total, 10) : receivedItems.length;
			var totalPages = (p && p.total_pages) ? parseInt(p.total_pages, 10) : 1;
			var curPage = (p && p.page) ? parseInt(p.page, 10) : page;
			var shown = receivedItems.length;
			var hasMore = curPage < totalPages;
			var btn = hasMore ? '<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-load-more" ' + (loadingMore ? 'disabled' : '') + '>' + escHtml(t('btn_load_more')) + '</button>' : '';
			$('#rsc-pager').html(
				'<div class="rsc-pager__row">' +
					'<div class="rsc-muted">' + escHtml(t('label_showing')) + ' <strong>' + escHtml(String(shown)) + '</strong> ' + escHtml(t('label_of')) + ' <strong>' + escHtml(String(total)) + '</strong></div>' +
					'<div>' + btn + '</div>' +
				'</div>'
			);
		}

		function load(opts){
			opts = opts || {};
			var force = !!opts.force;
			var reset = !!opts.reset;
			var append = !!opts.append;
			var searchVal = $('#rsc-search').val() || '';
			var catVal = $('#rsc-category').val() || '';
			var qKey = searchVal + '|' + catVal;

			if(reset || qKey !== lastQueryKey){
				page = 1;
				receivedItems = [];
				lastQueryKey = qKey;
				$('#rsc-items').html('<div class="rsc-muted">' + escHtml(t('msg_loading')) + '</div>');
				$('#rsc-pager').empty();
			}

			if(append){
				loadingMore = true;
				renderPager({ page: page, total_pages: page + 1, total: receivedItems.length + 1 });
			}

			ajax('rs_client_list_marketplace', {
				type: type,
				search: searchVal,
				category: catVal,
				page: page,
				per_page: perPage,
				force: force ? 1 : 0
			}).done(function(res){
				loadingMore = false;
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				access = res.data.access || {};
				renderMarketNotice();
				renderCategories(res.data.categories || []);

				var items = res.data.items || [];
				var p = res.data.pagination || {};
				if(reset || receivedItems.length===0){
					receivedItems = items.slice(0);
					renderItems(items, false);
				} else {
					receivedItems = receivedItems.concat(items);
					renderItems(items, true);
				}
				renderPager(p);

				// Background revalidate when we served from transient cache.
				if(res.data.from_cache && !force && !revalidateInFlight && page === 1){
					revalidateInFlight = true;
					setTimeout(function(){
						load({ reset:true, force:true });
						revalidateInFlight = false;
					}, 250);
				}
			}).fail(function(xhr){ loadingMore = false; notify('error', ajaxError(xhr)); });
		}

		load({ reset:true, force:false });

		$(document).off('click','#rsc-refresh').on('click','#rsc-refresh', function(){ load({ reset:true, force:true }); });
		$(document).off('keyup','#rsc-search').on('keyup','#rsc-search', function(){
			clearTimeout(debounceTimer);
			debounceTimer = setTimeout(function(){ load({ reset:true, force:false }); }, 350);
		});
		$(document).off('change','#rsc-category').on('change','#rsc-category', function(){ load({ reset:true, force:false }); });
		$(document).off('click','#rsc-load-more').on('click','#rsc-load-more', function(){
			page = page + 1;
			load({ append:true, force:false });
		});

		function setBtnLoading($btn, isLoading, loadingText){
			if(!$btn || !$btn.length){ return; }
			if(isLoading){
				$btn.data('orig-text', $btn.text());
				$btn.prop('disabled', true).addClass('rsc-btn--loading');
				$btn.text(loadingText || t('msg_loading'));
			} else {
				$btn.prop('disabled', false).removeClass('rsc-btn--loading');
				var ot = $btn.data('orig-text');
				if(ot){ $btn.text(ot); }
			}
		}

		$(document).off('click','.rsc-install').on('click','.rsc-install', function(){
			var $btn = $(this);
			var slug = $btn.data('slug');
			var ttype = $(this).data('type');
			setBtnLoading($btn, true, t('msg_installing'));
			ajax('rs_client_install', {slug: slug, type: ttype}).done(function(res){
				if(!res || !res.success){
					setBtnLoading($btn, false);
					notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error'));
					if(res && res.data && res.data.action_url){ window.open(res.data.action_url, '_blank'); }
					return;
				}
				notify('success', t('msg_done'));
				load({ reset:true, force:true });
			}).fail(function(xhr){ setBtnLoading($btn, false); notify('error', ajaxError(xhr)); });
		});

		$(document).off('click','.rsc-update').on('click','.rsc-update', function(){
			var $btn = $(this);
			var slug = $btn.data('slug');
			var ttype = $(this).data('type');
			var it = itemIndex[slug] || {};
			var newVersion = it.version || '';
			setBtnLoading($btn, true, t('msg_updating'));
			ajax('rs_client_update', {slug: slug, type: ttype, new_version: newVersion}).done(function(res){
				setBtnLoading($btn, false);
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				notify('success', t('msg_done'));
				load({ reset:true, force:true });
			}).fail(function(xhr){ setBtnLoading($btn, false); notify('error', ajaxError(xhr)); });
		});

		$(document).off('click','.rsc-activate').on('click','.rsc-activate', function(){
			var $btn = $(this);
			var slug = $btn.data('slug');
			setBtnLoading($btn, true, t('msg_loading'));
			ajax('rs_client_activate', {slug: slug}).done(function(res){
				setBtnLoading($btn, false);
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				notify('success', t('msg_done'));
				load({ reset:true, force:true });
			}).fail(function(xhr){ setBtnLoading($btn, false); notify('error', ajaxError(xhr)); });
		});

		$(document).off('click','.rsc-preview').on('click','.rsc-preview', function(){
			var slug = $(this).data('slug');
			var item = itemIndex[slug] || {};
			var body = '';
			if(item.screenshot_url){
				body += '<div style="margin-bottom:12px;"><img src="' + escHtml(item.screenshot_url) + '" alt="" style="width:100%;height:auto;border-radius:14px;border:1px solid rgba(0,0,0,.08);" /></div>';
			} else if(item.image_url){
				body += '<div style="margin-bottom:12px;"><img src="' + escHtml(item.image_url) + '" alt="" style="width:100%;height:auto;border-radius:14px;border:1px solid rgba(0,0,0,.08);" /></div>';
			}
			if(item.description){
				body += '<div class="rsc-item__desc" style="margin-bottom:12px;">' + escHtml(item.description) + '</div>';
			}
			if(item.changelog){
				body += '<div class="rsc-card__label" style="margin-bottom:8px;">' + escHtml(t('label_changelog')) + '</div>';
				body += '<div class="rsc-item__desc" style="white-space:pre-wrap;">' + escHtml(item.changelog) + '</div>';
			}
			if(item.demo_url){
				body += '<div style="margin-top:14px;"><a class="rsc-btn" href="' + escHtml(item.demo_url) + '" target="_blank" rel="noopener">' + escHtml(t('btn_demo')) + '</a></div>';
			}
			openModal(item.name || item.slug || t('btn_preview'), body, '');
		});
	}

	function renderUpdates(){
		var $c = shell(t('nav_updates'),
			'<button type="button" class="rsc-btn" id="rsc-update-all" style="display:none;">' + escHtml(t('btn_update_all')) + '</button>' +
			'<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-refresh">' + escHtml(t('btn_refresh')) + '</button>'
		);

		$c.html('<div class="rsc-toolbar" id="rsc-updates-toolbar"></div><div id="rsc-updates"></div>');

		var state = { installed: {plugins:[],themes:[]}, updates: {plugins:[],themes:[]} };
		var index = {};

		function renderInstalledCard(item){
			index[item.type + ':' + item.slug] = item;
			var actions = '';
			if(item.update_available){
				actions += '<button class="rsc-btn rsc-update-one" data-type="' + escHtml(item.type) + '" data-slug="' + escHtml(item.slug) + '">' + escHtml(t('btn_update')) + '</button>';
			}
			if(item.demo_url){
				actions += '<a class="rsc-btn rsc-btn--ghost" href="' + escHtml(item.demo_url) + '" target="_blank" rel="noopener">' + escHtml(t('btn_demo')) + '</a>';
			}
			var canPreview = (item.type === 'theme') && (item.screenshot_url || item.image_url || item.changelog || item.description);
			if(canPreview){
				actions += '<button class="rsc-btn rsc-btn--ghost rsc-preview" data-slug="' + escHtml(item.slug) + '" data-type="' + escHtml(item.type) + '">' + escHtml(t('btn_preview')) + '</button>';
			}

			var meta = '';
			meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_current')) + ': <strong>' + escHtml(item.current || '-') + '</strong></span>';
			meta += '<span class="rsc-item__metaPill">' + escHtml(t('label_new')) + ': <strong>' + escHtml(item.new || '-') + '</strong></span>';
			if(item.update_available){
				meta += '<span class="rsc-item__metaPill rsc-item__metaPill--update">' + escHtml(t('btn_update')) + '</span>';
			}

			var thumbUrl = '';
			if(item.type === 'theme'){
				thumbUrl = item.screenshot_url || item.image_url || '';
			} else {
				thumbUrl = item.image_url || '';
			}
			var thumb = '';
			if(thumbUrl){
				thumb = '<img class="rsc-item__img" src="' + escHtml(thumbUrl) + '" alt="" loading="lazy" />';
			} else {
				thumb = '<div class="rsc-item__imgPlaceholder">' + escHtml((item.name||item.slug||'').substring(0,1).toUpperCase() || 'R') + '</div>';
			}

			return '' +
				'<div class="rsc-item">' +
					'<div class="rsc-item__media">' + thumb + '</div>' +
					'<div class="rsc-item__body">' +
						'<div class="rsc-item__header">' +
							'<div>' +
								'<div class="rsc-item__titleRow">' +
									'<h3 class="rsc-item__name">' + escHtml(item.name || item.slug) + '</h3>' +
									'<div class="rsc-item__badges">' + badge(item.access_level) + '</div>' +
								'</div>' +
								'<div class="rsc-item__metaRow">' + meta + '</div>' +
							'</div>' +
						'</div>' +
						(item.description ? '<div class="rsc-item__desc">' + escHtml(item.description) + '</div>' : '') +
						'<div class="rsc-item__actions">' + actions + '</div>' +
					'</div>' +
				'</div>';
		}

		function renderSection(label, rows){
			var html = '<div class="rsc-card"><div class="rsc-card__label">' + escHtml(label) + '</div>';
			if(!rows || !rows.length){
				html += '<div class="rsc-muted">' + escHtml(t('msg_no_data')) + '</div></div>';
				return html;
			}
			html += '<div class="rsc-cards">';
			rows.forEach(function(r){ html += renderInstalledCard(r); });
			html += '</div></div>';
			return html;
		}

		function renderAll(){
			index = {};
			var pluginUpdates = (state.updates && state.updates.plugins) ? state.updates.plugins : [];
			var themeUpdates  = (state.updates && state.updates.themes) ? state.updates.themes : [];
			var totalUpdates = pluginUpdates.length + themeUpdates.length;

			if(totalUpdates <= 0){
				$('#rsc-updates').html(
					'<div class="rsc-card">' +
						'<div class="rsc-card__label">' + escHtml(t('nav_updates')) + '</div>' +
						'<div class="rsc-muted">' + escHtml(t('msg_no_data')) + '</div>' +
					'</div>'
				);
				$('#rsc-update-all').hide();
				return;
			}

			var html = '';
			if(pluginUpdates.length){
				html += renderSection(t('nav_plugins'), pluginUpdates);
			}
			if(themeUpdates.length){
				html += (html ? '<div style="margin-top:14px;">' : '') + renderSection(t('nav_themes'), themeUpdates) + (html ? '</div>' : '');
			}
			$('#rsc-updates').html(html);
			$('#rsc-update-all').show();
		}

		function renderAutoUpdateToolbar(data){
			data = data || {};
			var access = data.access || {};
			var allowed = (access && access.access_level === 'pro');
			var enabled = !!data.enabled;
			var toggleCls = allowed ? '' : ' rsc-toggle--disabled';
			var checked = enabled ? 'checked' : '';
			var disabled = allowed ? '' : 'disabled';
			var upgradeBtn = '';
			if(!allowed && access && access.action_url){
				upgradeBtn = '<a class="rsc-btn" href="' + escHtml(access.action_url) + '" target="_blank" rel="noopener">' + escHtml((access.reason==='domain_not_registered') ? t('cta_register_domain') : t('cta_upgrade')) + '</a>';
			}
			$('#rsc-updates-toolbar').html(
				'<label class="rsc-toggle' + toggleCls + '">' +
					'<input type="checkbox" id="rsc-auto-update" ' + checked + ' ' + disabled + ' />' +
					'<span class="rsc-toggle__slider"></span>' +
					'<span>' + escHtml(t('label_auto_update')) + '</span>' +
				'</label>' +
				upgradeBtn
			);
		}

		function load(){
			$('#rsc-updates').html('<div class="rsc-muted">' + escHtml(t('msg_loading')) + '</div>');
			$('#rsc-updates-toolbar').html('<div class="rsc-muted">' + escHtml(t('msg_loading')) + '</div>');
			ajax('rs_client_get_updates', {}).done(function(res){
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				state = res.data || state;
				renderAll();
				ajax('rs_client_get_auto_update', {}).done(function(r2){
					if(r2 && r2.success){ renderAutoUpdateToolbar(r2.data || {}); }
					else { renderAutoUpdateToolbar({enabled:false, access:{}}); }
				}).fail(function(){ renderAutoUpdateToolbar({enabled:false, access:{}}); });
			}).fail(function(xhr){ notify('error', ajaxError(xhr)); });
		}

		function setBtnLoading($btn, isLoading, loadingText){
			if(!$btn || !$btn.length){ return; }
			if(isLoading){
				$btn.data('orig-text', $btn.text());
				$btn.prop('disabled', true).addClass('rsc-btn--loading');
				$btn.text(loadingText || t('msg_loading'));
			} else {
				$btn.prop('disabled', false).removeClass('rsc-btn--loading');
				var ot = $btn.data('orig-text');
				if(ot){ $btn.text(ot); }
			}
		}

		load();
		$(document).off('click','#rsc-refresh').on('click','#rsc-refresh', load);

		$(document).off('change','#rsc-auto-update').on('change','#rsc-auto-update', function(){
			var $cb = $(this);
			var desired = $cb.is(':checked') ? 1 : 0;
			$cb.prop('disabled', true);
			ajax('rs_client_set_auto_update', { enabled: desired }).done(function(res){
				$cb.prop('disabled', false);
				if(!res || !res.success){
					$cb.prop('checked', !desired);
					notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error'));
					return;
				}
				notify('success', t('msg_done'));
			}).fail(function(xhr){
				$cb.prop('disabled', false);
				$cb.prop('checked', !desired);
				notify('error', ajaxError(xhr));
			});
		});

		$(document).off('click','.rsc-update-one').on('click','.rsc-update-one', function(){
			var $btn = $(this);
			var slug = $btn.data('slug');
			var ttype = $btn.data('type');
			var it = index[ttype + ':' + slug] || {};
			setBtnLoading($btn, true, t('msg_updating'));
			ajax('rs_client_update', {slug: slug, type: ttype, new_version: it.new || ''}).done(function(res){
				setBtnLoading($btn, false);
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				notify('success', t('msg_done'));
				load();
			}).fail(function(xhr){ setBtnLoading($btn, false); notify('error', ajaxError(xhr)); });
		});

		$(document).off('click','#rsc-update-all').on('click','#rsc-update-all', function(){
			var $btn = $(this);
			if(!confirm(t('msg_confirm'))){ return; }
			var items = [];
			((state.updates && state.updates.plugins) ? state.updates.plugins : []).forEach(function(r){ items.push({type:'plugin', slug:r.slug, new_version:r.new}); });
			((state.updates && state.updates.themes) ? state.updates.themes : []).forEach(function(r){ items.push({type:'theme', slug:r.slug, new_version:r.new}); });
			if(!items.length){ notify('success', t('msg_no_data')); return; }
			setBtnLoading($btn, true, t('msg_updating'));
			ajax('rs_client_update_all', {items: items}).done(function(res){
				setBtnLoading($btn, false);
				if(!res || !res.success){ notify('error', (res && res.data && res.data.message) ? res.data.message : t('msg_error')); return; }
				notify('success', t('msg_done'));
				load();
			}).fail(function(xhr){ setBtnLoading($btn, false); notify('error', ajaxError(xhr)); });
		});

		$(document).off('click','.rsc-preview').on('click','.rsc-preview', function(){
			var slug = $(this).data('slug');
			var ttype = $(this).data('type') || 'theme';
			var it = index[ttype + ':' + slug] || {};
			var body = '';
			if(it.screenshot_url){
				body += '<div style="margin-bottom:12px;"><img src="' + escHtml(it.screenshot_url) + '" alt="" style="width:100%;height:auto;border-radius:14px;border:1px solid rgba(0,0,0,.08);" /></div>';
			} else if(it.image_url){
				body += '<div style="margin-bottom:12px;"><img src="' + escHtml(it.image_url) + '" alt="" style="width:100%;height:auto;border-radius:14px;border:1px solid rgba(0,0,0,.08);" /></div>';
			}
			if(it.description){
				body += '<div class="rsc-item__desc" style="margin-bottom:12px;">' + escHtml(it.description) + '</div>';
			}
			if(it.changelog){
				body += '<div class="rsc-card__label" style="margin-bottom:8px;">' + escHtml(t('label_changelog')) + '</div>';
				body += '<div class="rsc-item__desc" style="white-space:pre-wrap;">' + escHtml(it.changelog) + '</div>';
			}
			if(it.demo_url){
				body += '<div style="margin-top:14px;"><a class="rsc-btn" href="' + escHtml(it.demo_url) + '" target="_blank" rel="noopener">' + escHtml(t('btn_demo')) + '</a></div>';
			}
			openModal(it.name || it.slug || t('btn_preview'), body, '');
		});
	}

	function renderNotifications(){
		var $c = shell(t('nav_notifications'),
			'<button type="button" class="rsc-btn rsc-btn--ghost" id="rsc-refresh">' + escHtml(t('btn_refresh')) + '</button>'
		);

		$c.html('<div id="rsc-notifs"></div>');

		function load(){
			$('#rsc-notifs').html('<div class="rsc-muted">' + escHtml(t('msg_loading')) + '</div>');
			ajax('rs_client_get_access', {}).done(function(res){
				if(!res || !res.success){ $('#rsc-notifs').html('<div class="rsc-muted">' + escHtml(t('msg_error')) + '</div>'); return; }
				var access = res.data.access || {};
				var items = [];

				if(access.ip_based_access){
					items.push({type:'info', message: t('msg_hosting_partner')});
				}

				if(access.reason === 'domain_not_registered'){
					items.push({type:'warning', message: t('msg_domain_required')});
				}

				if(access.access_level !== 'pro' && access.reason && access.reason !== 'domain_not_registered'){
					items.push({type:'warning', message: t('msg_access_denied')});
				}

				if(!items.length){
					$('#rsc-notifs').html('<div class="rsc-muted">' + escHtml(t('msg_no_data')) + '</div>');
					return;
				}

				var html = '';
				items.forEach(function(it){
					html += '<div class="rsc-card"><div class="rsc-card__label">' + escHtml(t('nav_notifications')) + '</div><div class="rsc-item__desc">' + escHtml(it.message) + '</div></div>';
				});
				$('#rsc-notifs').html(html);
			}).fail(function(){ $('#rsc-notifs').html('<div class="rsc-muted">' + escHtml(t('msg_error')) + '</div>'); });
		}

		load();
		$(document).off('click','#rsc-refresh').on('click','#rsc-refresh', load);
	}


	$(function(){
		var $app = $('#rs-client-app');
		if(!$app.length){ return; }

		var view = $app.data('view');

		var map = {
			'pluxix-client': renderDashboard,
			'pluxix-client-plugins': function(){ renderMarketplace('plugin'); },
			'pluxix-client-themes': function(){ renderMarketplace('theme'); },
			'pluxix-client-updates': renderUpdates,
			'pluxix-client-notifications': renderNotifications
		};

		if(map[view]){ map[view](); } else { renderDashboard(); }
	});
})(jQuery);
