<?php

namespace pluxix\Client\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Admin {
	private static $instance = null;

	public static function instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'admin_menu', array( $this, 'register_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	public function register_menu(): void {
		$cap = 'manage_options';

		add_menu_page(
			esc_html__( 'PLUXIX', 'pluxix-client' ),
			esc_html__( 'PLUXIX', 'pluxix-client' ),
			$cap,
			'pluxix-client',
			array( $this, 'render_app' ),
			'dashicons-store',
			56
		);

		$pages = array(
			'pluxix-client'               => esc_html__( 'Dashboard', 'pluxix-client' ),
			'pluxix-client-plugins'       => esc_html__( 'Plugins', 'pluxix-client' ),
			'pluxix-client-themes'        => esc_html__( 'Themes', 'pluxix-client' ),
			'pluxix-client-updates'       => esc_html__( 'Updates', 'pluxix-client' ),
			'pluxix-client-notifications' => esc_html__( 'Notifications', 'pluxix-client' ),
		);

		foreach ( $pages as $slug => $title ) {
			add_submenu_page(
				'pluxix-client',
				$title,
				$title,
				$cap,
				$slug,
				array( $this, 'render_app' )
			);
		}
	}

	public function enqueue_assets( string $hook ): void {
		if ( empty( $_GET['page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}

		$page = sanitize_key( wp_unslash( $_GET['page'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( 0 !== strpos( $page, 'pluxix-client' ) ) {
			return;
		}

		wp_enqueue_style(
			'pluxix-client-admin',
			pluxix_CLIENT_URL . 'assets/admin/admin.css',
			array(),
			pluxix_CLIENT_VERSION
		);

		wp_enqueue_script(
			'pluxix-client-admin',
			pluxix_CLIENT_URL . 'assets/admin/app.js',
			array( 'jquery' ),
			pluxix_CLIENT_VERSION,
			true
		);

		$domain = wp_parse_url( home_url(), PHP_URL_HOST );
		if ( ! $domain ) {
			$domain = '';
		}

		wp_localize_script(
			'pluxix-client-admin',
			'pluxixClient',
			array(
				'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
				'nonce'     => wp_create_nonce( 'pluxix_client_admin' ),
				'isRTL'     => is_rtl(),
				'siteDomain'=> $domain,
				'i18n'      => $this->get_i18n_strings(),
			)
		);
	}

	public function render_app(): void {
		$page = isset( $_GET['page'] ) ? sanitize_key( wp_unslash( $_GET['page'] ) ) : 'pluxix-client'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		?>
		<div class="wrap">
			<div id="rs-client-app" class="rs-app" data-view="<?php echo esc_attr( $page ); ?>">
				<noscript><?php echo esc_html__( 'JavaScript is required to use pluxix Client.', 'pluxix-client' ); ?></noscript>
			</div>
		</div>
		<?php
	}

	private function get_i18n_strings(): array {
		return array(
			'nav_dashboard'       => esc_html__( 'Dashboard', 'pluxix-client' ),
			'nav_plugins'         => esc_html__( 'Plugins Marketplace', 'pluxix-client' ),
			'nav_themes'          => esc_html__( 'Themes Marketplace', 'pluxix-client' ),
			'nav_updates'         => esc_html__( 'Updates Center', 'pluxix-client' ),
			'nav_notifications'   => esc_html__( 'Notifications', 'pluxix-client' ),

			'notice_free_title'   => esc_html__( 'You are using the Free plan', 'pluxix-client' ),
			'notice_free_text'    => esc_html__( 'Upgrade to Pro to unlock all items and enable Auto Update.', 'pluxix-client' ),


			'label_access'        => esc_html__( 'Access', 'pluxix-client' ),
			'label_access_free'   => esc_html__( 'Free', 'pluxix-client' ),
			'label_access_pro'    => esc_html__( 'Pro', 'pluxix-client' ),
			'label_access_type'   => esc_html__( 'Access Type', 'pluxix-client' ),
			'label_subscription'  => esc_html__( 'Subscription', 'pluxix-client' ),
			'label_manual'        => esc_html__( 'Manual', 'pluxix-client' ),
			'label_hosting_partner'=> esc_html__( 'Hosting Partner', 'pluxix-client' ),
			'label_domain'        => esc_html__( 'Domain', 'pluxix-client' ),
			'label_status'        => esc_html__( 'Status', 'pluxix-client' ),
			'label_end_date'      => esc_html__( 'End Date', 'pluxix-client' ),
			'label_domain_status' => esc_html__( 'Domain Status', 'pluxix-client' ),

			'label_version'       => esc_html__( 'Version', 'pluxix-client' ),
			'label_installed'     => esc_html__( 'Installed', 'pluxix-client' ),
			'label_active'        => esc_html__( 'Active', 'pluxix-client' ),
			'label_inactive'      => esc_html__( 'Inactive', 'pluxix-client' ),
			'label_not_installed' => esc_html__( 'Not installed', 'pluxix-client' ),
			'label_current'       => esc_html__( 'Current', 'pluxix-client' ),
			'label_new'           => esc_html__( 'New', 'pluxix-client' ),
			'label_actions'       => esc_html__( 'Actions', 'pluxix-client' ),
			'label_changelog'    => esc_html__( 'Changelog', 'pluxix-client' ),
			'label_auto_update'  => esc_html__( 'Auto Update (Weekly)', 'pluxix-client' ),


			'badge_free'          => esc_html__( 'FREE', 'pluxix-client' ),
			'badge_pro'           => esc_html__( 'PRO', 'pluxix-client' ),

			'btn_refresh'         => esc_html__( 'Refresh', 'pluxix-client' ),
			'btn_recheck_access'  => esc_html__( 'Recheck Access', 'pluxix-client' ),
			'btn_sync'            => esc_html__( 'Manual Sync', 'pluxix-client' ),
			'btn_clear_cache'     => esc_html__( 'Clear Cache', 'pluxix-client' ),
			'btn_save'            => esc_html__( 'Save', 'pluxix-client' ),
			'btn_cancel'          => esc_html__( 'Cancel', 'pluxix-client' ),
			'btn_close'           => esc_html__( 'Close', 'pluxix-client' ),

			'btn_install'         => esc_html__( 'Install', 'pluxix-client' ),
			'btn_activate'        => esc_html__( 'Activate', 'pluxix-client' ),
			'btn_update'          => esc_html__( 'Update', 'pluxix-client' ),
			'btn_update_all'      => esc_html__( 'Update All', 'pluxix-client' ),
			'btn_preview'        => esc_html__( 'Preview', 'pluxix-client' ),

			'label_search'        => esc_html__( 'Search...', 'pluxix-client' ),
			'label_filter_category'=> esc_html__( 'Filter by category', 'pluxix-client' ),
			'label_all'           => esc_html__( 'All', 'pluxix-client' ),

			'msg_loading'         => esc_html__( 'Loading...', 'pluxix-client' ),
			'msg_error'           => esc_html__( 'Something went wrong.', 'pluxix-client' ),
			'msg_saved'           => esc_html__( 'Saved.', 'pluxix-client' ),
			'msg_installing'      => esc_html__( 'Installing...', 'pluxix-client' ),
			'msg_updating'        => esc_html__( 'Updating...', 'pluxix-client' ),
			'msg_done'            => esc_html__( 'Done.', 'pluxix-client' ),
			'msg_no_data'         => esc_html__( 'No data found.', 'pluxix-client' ),
			'msg_confirm'         => esc_html__( 'Are you sure?', 'pluxix-client' ),
			'msg_access_denied'   => esc_html__( 'Access denied.', 'pluxix-client' ),
			'msg_domain_required' => esc_html__( 'To use Pro, you must register your domain on the server.', 'pluxix-client' ),
			'msg_hosting_partner' => esc_html__( 'Because you are using one of our partner hosting providers, you can use the Pro version for free.', 'pluxix-client' ),

			'btn_demo'            => esc_html__( 'Demo', 'pluxix-client' ),
			'btn_load_more'       => esc_html__( 'Load More', 'pluxix-client' ),
			'label_showing'       => esc_html__( 'Showing', 'pluxix-client' ),
			'label_of'            => esc_html__( 'of', 'pluxix-client' ),

			'cta_upgrade'         => esc_html__( 'Upgrade to Pro', 'pluxix-client' ),
			'cta_register_domain' => esc_html__( 'Register Domain', 'pluxix-client' ),
		);
	}
}
