<?php

namespace pluxix\Client\Admin;

use pluxix\Client\Api\Server_Api;
use pluxix\Client\Services\Package_Installer;
use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Ajax {
	private static $instance = null;

	public static function instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_action( 'wp_ajax_rs_client_get_access', array( $this, 'get_access' ) );
		add_action( 'wp_ajax_rs_client_list_marketplace', array( $this, 'list_marketplace' ) );
		add_action( 'wp_ajax_rs_client_install', array( $this, 'install' ) );
		add_action( 'wp_ajax_rs_client_activate', array( $this, 'activate' ) );
		add_action( 'wp_ajax_rs_client_update', array( $this, 'update' ) );
		add_action( 'wp_ajax_rs_client_get_updates', array( $this, 'get_updates' ) );
		add_action( 'wp_ajax_rs_client_update_all', array( $this, 'update_all' ) );
		add_action( 'wp_ajax_rs_client_get_auto_update', array( $this, 'get_auto_update' ) );
		add_action( 'wp_ajax_rs_client_set_auto_update', array( $this, 'set_auto_update' ) );
		add_action( 'wp_ajax_rs_client_clear_cache', array( $this, 'clear_cache' ) );
		add_action( 'wp_ajax_rs_client_sync', array( $this, 'sync' ) );
	}

	private function verify(): void {
		check_ajax_referer( 'pluxix_client_admin', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array( 'message' => esc_html__( 'Access denied.', 'pluxix-client' ) ),
				403
			);
		}
	}

	private function cache_key( string $type, string $search = '', int $category = 0, int $page = 1, int $per_page = 0 ): string {
		return 'rs_client_' . $type . '_' . md5( $search . '|' . $category . '|' . (int) $page . '|' . (int) $per_page );
	}

	private function remember_cache_key( string $key ): void {
		$keys = get_option( 'pluxix_client_cache_keys', array() );
		if ( ! is_array( $keys ) ) {
			$keys = array();
		}
		if ( ! in_array( $key, $keys, true ) ) {
			$keys[] = $key;
			update_option( 'pluxix_client_cache_keys', $keys );
		}
	}

	private function get_map(): array {
		$map = get_option( 'pluxix_client_map', array() );
		$defaults = array(
			'plugins' => array(),
			'themes'  => array(),
			'theme_children' => array(),
		);
		if ( ! is_array( $map ) ) {
			$map = array();
		}
		$map = array_merge( $defaults, $map );
		foreach ( array( 'plugins', 'themes', 'theme_children' ) as $k ) {
			if ( ! isset( $map[ $k ] ) || ! is_array( $map[ $k ] ) ) {
				$map[ $k ] = array();
			}
		}
		return $map;
	}

	private function update_map( array $map ): void {
		update_option( 'pluxix_client_map', $map );
	}

	private function find_plugin_file_by_slug( string $slug ): string {
		$map = $this->get_map();

		if ( isset( $map['plugins'][ $slug ] ) && is_string( $map['plugins'][ $slug ] ) && '' !== $map['plugins'][ $slug ] ) {
			return $map['plugins'][ $slug ];
		}

		// Guess common pattern: slug/slug.php.
		$guess = $slug . '/' . $slug . '.php';
		if ( file_exists( WP_PLUGIN_DIR . '/' . $guess ) ) {
			return $guess;
		}

		// Fallback: scan installed plugins once.
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		$plugins = get_plugins();
		foreach ( $plugins as $file => $data ) {
			if ( 0 === strpos( $file, $slug . '/' ) ) {
				return (string) $file;
			}
		}

		return '';
	}

	private function get_plugin_state( string $slug ): array {
		$plugin_file = $this->find_plugin_file_by_slug( $slug );
		$installed   = ( '' !== $plugin_file ) && file_exists( WP_PLUGIN_DIR . '/' . $plugin_file );

		$active = false;
		$ver    = '';

		if ( $installed ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
			$active = is_plugin_active( $plugin_file );

			$plugins = get_plugins();
			if ( isset( $plugins[ $plugin_file ]['Version'] ) ) {
				$ver = (string) $plugins[ $plugin_file ]['Version'];
			}
		}

		return array(
			'installed'   => $installed,
			'active'      => $active,
			'plugin_file' => $plugin_file,
			'version'     => $ver,
		);
	}

	private function get_theme_state( string $slug ): array {
		$map = $this->get_map();
		$stylesheet = '';
		if ( isset( $map['themes'][ $slug ] ) && is_string( $map['themes'][ $slug ] ) && '' !== $map['themes'][ $slug ] ) {
			$stylesheet = $map['themes'][ $slug ];
		} else {
			$stylesheet = $slug;
		}

		$theme = wp_get_theme( $stylesheet );
		$installed = $theme && $theme->exists();
		$ver = $installed ? (string) $theme->get( 'Version' ) : '';

		return array(
			'installed'  => $installed,
			'stylesheet' => $stylesheet,
			'version'    => $ver,
		);
	}

	

	private function find_child_theme_stylesheet( string $parent_stylesheet ): string {
		$parent_stylesheet = sanitize_key( $parent_stylesheet );
		if ( '' === $parent_stylesheet ) {
			return '';
		}

		$themes = wp_get_themes();
		foreach ( $themes as $stylesheet => $theme ) {
			if ( ! $theme || ! $theme->exists() ) {
				continue;
			}
			$template = (string) $theme->get( 'Template' );
			if ( '' !== $template && $template === $parent_stylesheet ) {
				return (string) $stylesheet;
			}
		}

		return '';
	}

	private function sync_theme_bundles( string $slug, array $access, array &$map ): array {
		$bundle = array(
			'child'    => array(),
			'plugins'  => array(),
			'warnings' => array(),
		);

		$remote_item   = $this->get_remote_item_cached( 'theme', $slug );
		$bundle_slugs  = array();
		$has_child     = false;

		if ( is_array( $remote_item ) ) {
			if ( ! empty( $remote_item['bundle_plugins'] ) && is_array( $remote_item['bundle_plugins'] ) ) {
				$bundle_slugs = $remote_item['bundle_plugins'];
			}
			$has_child = ! empty( $remote_item['has_child_theme'] );
		}

		// Child theme sync (install if missing, update when present).
		if ( $has_child ) {
			$child_url = Server_Api::get_theme_child_download_url( $slug );
			if ( '' === $child_url ) {
				$bundle['warnings'][] = esc_html__( 'Child theme is enabled on server but no download URL was returned.', 'pluxix-client' );
			} else {
				$parent_state      = $this->get_theme_state( $slug );
				$parent_stylesheet = isset( $parent_state['stylesheet'] ) ? (string) $parent_state['stylesheet'] : '';

				$child_stylesheet = '';
				if ( isset( $map['theme_children'] ) && is_array( $map['theme_children'] ) && isset( $map['theme_children'][ $slug ] ) ) {
					$child_stylesheet = (string) $map['theme_children'][ $slug ];
				}
				if ( '' === $child_stylesheet && '' !== $parent_stylesheet ) {
					$child_stylesheet = $this->find_child_theme_stylesheet( $parent_stylesheet );
				}

				$child_installed = false;
				if ( '' !== $child_stylesheet ) {
					$child_theme     = wp_get_theme( $child_stylesheet );
					$child_installed = ( $child_theme && $child_theme->exists() );
				}

				if ( $child_installed ) {
					$child_res = Package_Installer::update_theme_from_package( $child_url, $child_stylesheet, $slug . '__child', '' );
					if ( is_wp_error( $child_res ) ) {
						$bundle['child'] = array( 'status' => 'error', 'message' => $child_res->get_error_message() );
					} else {
						$bundle['child'] = array( 'status' => 'updated', 'stylesheet' => $child_stylesheet );
					}
				} else {
					$child_res = Package_Installer::install_theme( $child_url );
					if ( is_wp_error( $child_res ) ) {
						$bundle['child'] = array( 'status' => 'error', 'message' => $child_res->get_error_message() );
					} else {
						$child_stylesheet = isset( $child_res['stylesheet'] ) ? (string) $child_res['stylesheet'] : '';
						if ( '' !== $child_stylesheet ) {
							if ( ! isset( $map['theme_children'] ) || ! is_array( $map['theme_children'] ) ) {
								$map['theme_children'] = array();
							}
							$map['theme_children'][ $slug ] = $child_stylesheet;
						}
						$bundle['child'] = array( 'status' => 'installed', 'stylesheet' => $child_stylesheet );
					}
				}
			}
		}

		// Bundled plugin sync (install if missing, update when newer).
		if ( is_array( $bundle_slugs ) && ! empty( $bundle_slugs ) ) {
			foreach ( $bundle_slugs as $pslug ) {
				$pslug = sanitize_key( (string) $pslug );
				if ( '' === $pslug ) {
					continue;
				}

				$item_access_p = $this->get_item_access_level( 'plugin', $pslug );
				$check_p       = $this->can_install_item( $access, $item_access_p );
				if ( isset( $check_p['ok'] ) && ! $check_p['ok'] ) {
					$bundle['plugins'][ $pslug ] = array(
						'status'     => 'skipped_no_access',
						'message'    => isset( $check_p['message'] ) ? $check_p['message'] : '',
						'action_url' => isset( $check_p['action_url'] ) ? $check_p['action_url'] : '',
					);
					continue;
				}

				$remote_p   = $this->get_remote_item_cached( 'plugin', $pslug );
				$remote_ver = ( is_array( $remote_p ) && isset( $remote_p['version'] ) ) ? (string) $remote_p['version'] : '';

				$p_download = Server_Api::get_download_url( 'plugin', $pslug );
				if ( '' === $p_download ) {
					$bundle['plugins'][ $pslug ] = array( 'status' => 'error', 'message' => esc_html__( 'Server URL is not configured.', 'pluxix-client' ) );
					continue;
				}

				$p_state = $this->get_plugin_state( $pslug );

				if ( empty( $p_state['installed'] ) ) {
					$ins = Package_Installer::install_plugin( $p_download );
					if ( is_wp_error( $ins ) ) {
						$bundle['plugins'][ $pslug ] = array( 'status' => 'error', 'message' => $ins->get_error_message() );
					} else {
						$plugin_file = isset( $ins['plugin_file'] ) ? (string) $ins['plugin_file'] : '';
						if ( '' !== $plugin_file ) {
							if ( ! isset( $map['plugins'] ) || ! is_array( $map['plugins'] ) ) {
								$map['plugins'] = array();
							}
							$map['plugins'][ $pslug ] = $plugin_file;
						}
						$bundle['plugins'][ $pslug ] = array( 'status' => 'installed' );
					}
					continue;
				}

				$current_ver  = isset( $p_state['version'] ) ? (string) $p_state['version'] : '';
				$needs_update = false;
				if ( '' !== $remote_ver && '' !== $current_ver ) {
					$needs_update = version_compare( $remote_ver, $current_ver, '>' );
				} elseif ( '' !== $remote_ver && '' === $current_ver ) {
					$needs_update = true;
				}

				if ( $needs_update ) {
					$plugin_file = isset( $p_state['plugin_file'] ) ? (string) $p_state['plugin_file'] : '';
					$upd         = Package_Installer::update_plugin_from_package( $p_download, $plugin_file, $pslug, $remote_ver ?: '0.0.0' );
					if ( is_wp_error( $upd ) ) {
						$bundle['plugins'][ $pslug ] = array( 'status' => 'error', 'message' => $upd->get_error_message() );
					} else {
						if ( is_array( $upd ) && ! empty( $upd['plugin_file'] ) ) {
							$map['plugins'][ $pslug ] = (string) $upd['plugin_file'];
						}
						$bundle['plugins'][ $pslug ] = array(
							'status'  => 'updated',
							'version' => ( is_array( $upd ) && isset( $upd['version'] ) ) ? (string) $upd['version'] : '',
						);
					}
				} else {
					$bundle['plugins'][ $pslug ] = array( 'status' => 'up_to_date' );
				}
			}
		}

		return $bundle;
	}

private function norm_ver( string $v ): string {
		$v = trim( $v );
		$v = preg_replace( '/^v/i', '', $v );
		$v = preg_replace( '/[^0-9.]/', '', $v );
		return is_string( $v ) ? $v : '';
	}

	private function ready_licenses_url(): string {
		$base = Server_Api::get_server_url();
		if ( '' === $base ) {
			return '';
		}
		// Centralized destination for upgrading / managing licenses.
		return $base . '/my-account/pluxix-store/';
	}

	private function force_access_action_url( array $access ): array {
		$url = $this->ready_licenses_url();
		if ( '' !== $url ) {
			$access['action_url'] = $url;
		}
		return $access;
	}

	private function get_remote_item_cached( string $type, string $slug ) {
		$type = ( 'theme' === $type ) ? 'theme' : 'plugin';
		$slug = sanitize_key( (string) $slug );
		if ( '' === $slug ) {
			return new WP_Error( 'pluxix_client_invalid_slug', esc_html__( 'Invalid slug.', 'pluxix-client' ) );
		}

		$ttl = Server_Api::get_cache_ttl();
		$key = 'rs_client_info_' . $type . '_' . md5( $slug );
		$cached = get_transient( $key );
		if ( is_array( $cached ) ) {
			return $cached;
		}

		$res = Server_Api::get_item_info( $type, $slug );
		if ( is_wp_error( $res ) ) {
			return $res;
		}

		// Server returns { success, item }.
		$item = isset( $res['item'] ) && is_array( $res['item'] ) ? $res['item'] : array();
		set_transient( $key, $item, $ttl );
		$this->remember_cache_key( $key );

		return $item;
	}

	public function get_access(): void {
		$this->verify();

		$ttl = Server_Api::get_cache_ttl();
		$key = 'rs_client_access';
		$cached = get_transient( $key );
		if ( is_array( $cached ) ) {
			$cached = $this->force_access_action_url( $cached );
			set_transient( $key, $cached, $ttl );
			wp_send_json_success( array( 'access' => $cached ) );
		}

		$res = Server_Api::check_access();
		if ( is_wp_error( $res ) ) {
			wp_send_json_error( array( 'message' => $res->get_error_message() ), 400 );
		}

		$access = isset( $res['access'] ) && is_array( $res['access'] ) ? $res['access'] : array();
		$access = $this->force_access_action_url( $access );

		set_transient( $key, $access, $ttl );
		update_option( 'pluxix_client_last_access', array( 'time' => time(), 'access' => $access ) );

		wp_send_json_success( array( 'access' => $access ) );
	}

	public function list_marketplace(): void {
		$this->verify();

		$type     = isset( $_POST['type'] ) ? sanitize_key( wp_unslash( $_POST['type'] ) ) : 'plugin';
		$search   = isset( $_POST['search'] ) ? sanitize_text_field( wp_unslash( $_POST['search'] ) ) : '';
		$category = isset( $_POST['category'] ) ? absint( wp_unslash( $_POST['category'] ) ) : 0;
		$page     = isset( $_POST['page'] ) ? max( 1, absint( wp_unslash( $_POST['page'] ) ) ) : 1;
		$per_page = isset( $_POST['per_page'] ) ? max( 1, min( 100, absint( wp_unslash( $_POST['per_page'] ) ) ) ) : 12;
		$force    = ! empty( $_POST['force'] );

		$type = ( 'theme' === $type ) ? 'theme' : 'plugin';

		$ttl  = Server_Api::get_cache_ttl();
		$key  = $this->cache_key( $type, $search, $category, $page, $per_page );

		$cached = get_transient( $key );
		if ( ! $force && is_array( $cached ) ) {
			if ( isset( $cached['access'] ) && is_array( $cached['access'] ) ) {
				$cached['access'] = $this->force_access_action_url( $cached['access'] );
			}
			$cached['from_cache'] = true;
			wp_send_json_success( $cached );
		}

		$res = Server_Api::get_marketplace( $type, $search, $category, $page, $per_page );
		if ( is_wp_error( $res ) ) {
			wp_send_json_error( array( 'message' => $res->get_error_message() ), 400 );
		}

		// Add local installation state.
		$items = isset( $res['items'] ) && is_array( $res['items'] ) ? $res['items'] : array();
		$pagination = isset( $res['pagination'] ) && is_array( $res['pagination'] ) ? $res['pagination'] : array();

		if ( empty( $pagination ) ) {
			$total = count( $items );
			$total_pages = ( $per_page > 0 ) ? (int) ceil( $total / $per_page ) : 1;
			$pagination = array(
				'page'        => (int) $page,
				'per_page'    => (int) $per_page,
				'total'       => (int) $total,
				'total_pages' => (int) max( 1, $total_pages ),
			);
			if ( $per_page > 0 && $total > $per_page ) {
				$offset = ( $page - 1 ) * $per_page;
				$items = array_slice( $items, $offset, $per_page );
			}
		}
		$out_items = array();

		foreach ( $items as $item ) {
			if ( ! is_array( $item ) || empty( $item['slug'] ) ) {
				continue;
			}

			$slug = (string) $item['slug'];

			$state = ( 'theme' === $type ) ? $this->get_theme_state( $slug ) : $this->get_plugin_state( $slug );

			$item['local_installed'] = (bool) $state['installed'];
			$item['local_version']   = (string) $state['version'];

			if ( 'theme' === $type ) {
				$item['local_stylesheet'] = (string) $state['stylesheet'];
			} else {
				$item['local_active']    = (bool) $state['active'];
				$item['local_plugin_file']= (string) $state['plugin_file'];
			}

			$remote_v = isset( $item['version'] ) ? $this->norm_ver( (string) $item['version'] ) : '';
			$local_v  = $this->norm_ver( (string) $state['version'] );
			$item['update_available'] = ( $state['installed'] && '' !== $remote_v && '' !== $local_v ) ? version_compare( $remote_v, $local_v, '>' ) : false;

			$out_items[] = $item;
		}

		$payload = array(
			'access'     => isset( $res['access'] ) && is_array( $res['access'] ) ? $this->force_access_action_url( $res['access'] ) : array(),
			'maintenance'=> isset( $res['maintenance'] ) ? (bool) $res['maintenance'] : false,
			'items'      => $out_items,
			'categories' => isset( $res['categories'] ) ? $res['categories'] : array(),
			'pagination' => $pagination,
			'from_cache' => false,
		);

		set_transient( $key, $payload, $ttl );
		$this->remember_cache_key( $key );

		wp_send_json_success( $payload );
	}

	private function get_item_access_level( string $type, string $slug ): string {
		$res = Server_Api::get_marketplace( $type, $slug, 0 );
		if ( is_wp_error( $res ) ) {
			return '';
		}
		$items = isset( $res['items'] ) && is_array( $res['items'] ) ? $res['items'] : array();
		foreach ( $items as $item ) {
			if ( is_array( $item ) && isset( $item['slug'] ) && (string) $item['slug'] === $slug ) {
				return isset( $item['access_level'] ) ? (string) $item['access_level'] : '';
			}
		}
		return '';
	}

	private function can_install_item( array $access, string $item_access_level ): array {
		$level = isset( $access['access_level'] ) ? (string) $access['access_level'] : 'free';

		if ( 'pro' === $item_access_level && 'pro' !== $level ) {
			$action_url = $this->ready_licenses_url();
			return array(
				'ok' => false,
				'message' => esc_html__( 'This item is available in Pro only.', 'pluxix-client' ),
				'action_url' => $action_url,
			);
		}

		return array( 'ok' => true );
	}

	public function install(): void {
		$this->verify();

		$type = isset( $_POST['type'] ) ? sanitize_key( wp_unslash( $_POST['type'] ) ) : 'plugin';
		$slug = isset( $_POST['slug'] ) ? sanitize_key( wp_unslash( $_POST['slug'] ) ) : '';

		if ( '' === $slug ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid slug.', 'pluxix-client' ) ), 400 );
		}

		$type = ( 'theme' === $type ) ? 'theme' : 'plugin';

		// Check access first for friendly error messages.
		$accessRes = Server_Api::check_access();
		if ( is_wp_error( $accessRes ) ) {
			wp_send_json_error( array( 'message' => $accessRes->get_error_message() ), 400 );
		}
		$access = isset( $accessRes['access'] ) ? $this->force_access_action_url( $accessRes['access'] ) : array();

		$item_access = $this->get_item_access_level( $type, $slug );
		$check = $this->can_install_item( $access, $item_access );
		if ( isset( $check['ok'] ) && ! $check['ok'] ) {
			wp_send_json_error(
				array(
					'message'    => $check['message'],
					'action_url' => isset( $check['action_url'] ) ? $check['action_url'] : '',
				),
				403
			);
		}

		$download_url = Server_Api::get_download_url( $type, $slug );
		if ( '' === $download_url ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Server URL is not configured.', 'pluxix-client' ) ), 400 );
		}

		$result = ( 'theme' === $type ) ? Package_Installer::install_theme( $download_url ) : Package_Installer::install_plugin( $download_url );

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ), 400 );
		}

		$map = $this->get_map();
		$bundle = array(
			'child'   => null,
			'plugins' => array(),
		);

		// Track installed package in map.
		if ( 'theme' === $type ) {
			if ( isset( $result['stylesheet'] ) && $result['stylesheet'] ) {
				$map['themes'][ $slug ] = (string) $result['stylesheet'];
			}
		} else {
			if ( isset( $result['plugin_file'] ) && $result['plugin_file'] ) {
				$map['plugins'][ $slug ] = (string) $result['plugin_file'];
			}
		}

		// Theme bundles: optional child theme + bundled marketplace plugins.
		if ( 'theme' === $type ) {
			$remote_item  = $this->get_remote_item_cached( 'theme', $slug );
			$bundle_slugs = array();
			$has_child    = false;
			if ( is_array( $remote_item ) ) {
				if ( ! empty( $remote_item['bundle_plugins'] ) && is_array( $remote_item['bundle_plugins'] ) ) {
					$bundle_slugs = $remote_item['bundle_plugins'];
				}
				$has_child = ! empty( $remote_item['has_child_theme'] );
			}

			// Install child theme (if provided on server).
			if ( $has_child ) {
				$child_url = Server_Api::get_theme_child_download_url( $slug );
				if ( '' !== $child_url ) {
					$child_res = Package_Installer::install_theme( $child_url );
					if ( is_wp_error( $child_res ) ) {
						$bundle['child'] = array( 'status' => 'error', 'message' => $child_res->get_error_message() );
					} else {
						$bundle['child'] = array( 'status' => 'installed', 'result' => $child_res );
						if ( is_array( $child_res ) && ! empty( $child_res['stylesheet'] ) ) {
							$map['theme_children'][ $slug ] = (string) $child_res['stylesheet'];
						}
					}
				} else {
					$bundle['child'] = array( 'status' => 'error', 'message' => esc_html__( 'Child theme download URL is missing.', 'pluxix-client' ) );
				}
			}

			// Install bundled plugins (from marketplace) by slug.
			if ( is_array( $bundle_slugs ) ) {
				foreach ( $bundle_slugs as $pslug ) {
					$pslug = sanitize_key( (string) $pslug );
					if ( '' === $pslug ) {
						continue;
					}
					$state = $this->get_plugin_state( $pslug );
					if ( ! empty( $state['installed'] ) ) {
						$bundle['plugins'][ $pslug ] = array( 'status' => 'already_installed' );
						continue;
					}

					$item_access_p = $this->get_item_access_level( 'plugin', $pslug );
					$check_p = $this->can_install_item( $access, $item_access_p );
					if ( isset( $check_p['ok'] ) && ! $check_p['ok'] ) {
						$bundle['plugins'][ $pslug ] = array(
							'status' => 'skipped_no_access',
							'message' => isset( $check_p['message'] ) ? $check_p['message'] : '',
							'action_url' => isset( $check_p['action_url'] ) ? $check_p['action_url'] : '',
						);
						continue;
					}

					$p_download = Server_Api::get_download_url( 'plugin', $pslug );
					if ( '' === $p_download ) {
						$bundle['plugins'][ $pslug ] = array( 'status' => 'error', 'message' => esc_html__( 'Server URL is not configured.', 'pluxix-client' ) );
						continue;
					}
					$p_result = Package_Installer::install_plugin( $p_download );
					if ( is_wp_error( $p_result ) ) {
						$bundle['plugins'][ $pslug ] = array( 'status' => 'error', 'message' => $p_result->get_error_message() );
						continue;
					}
					if ( is_array( $p_result ) && ! empty( $p_result['plugin_file'] ) ) {
						$map['plugins'][ $pslug ] = (string) $p_result['plugin_file'];
					}
					$bundle['plugins'][ $pslug ] = array( 'status' => 'installed', 'result' => $p_result );
				}
			}
		}

		$this->update_map( $map );

		// Telemetry: report installs + refresh heartbeat.
		$event_items = array();
		if ( 'theme' === $type ) {
			$t_state = $this->get_theme_state( $slug );
			$event_items[] = array(
				'product_type'      => 'theme',
				'slug'              => $slug,
				'identifier'        => isset( $t_state['stylesheet'] ) ? (string) $t_state['stylesheet'] : '',
				'installed_version' => isset( $t_state['version'] ) ? (string) $t_state['version'] : '',
				'is_active'         => ! empty( $t_state['active'] ) ? 1 : 0,
			);
			if ( isset( $bundle['child']['status'] ) && 'installed' === $bundle['child']['status'] && isset( $bundle['child']['result']['stylesheet'] ) ) {
				$c_stylesheet = (string) $bundle['child']['result']['stylesheet'];
				$c_theme = wp_get_theme( $c_stylesheet );
				$event_items[] = array(
					'product_type'      => 'theme_child',
					'slug'              => $slug,
					'identifier'        => $c_stylesheet,
					'installed_version' => $c_theme instanceof \WP_Theme ? (string) $c_theme->get( 'Version' ) : '',
					'is_active'         => ( $c_stylesheet === ( wp_get_theme() instanceof \WP_Theme ? wp_get_theme()->get_stylesheet() : '' ) ) ? 1 : 0,
				);
			}
			if ( ! empty( $bundle['plugins'] ) && is_array( $bundle['plugins'] ) ) {
				foreach ( $bundle['plugins'] as $pslug => $pdata ) {
					if ( isset( $pdata['status'] ) && 'installed' === $pdata['status'] && isset( $pdata['result']['plugin_file'] ) ) {
						$pfile = (string) $pdata['result']['plugin_file'];
						$pstate = $this->get_plugin_state( $pslug );
						$event_items[] = array(
							'product_type'      => 'plugin',
							'slug'              => (string) $pslug,
							'identifier'        => $pfile,
							'installed_version' => isset( $pstate['version'] ) ? (string) $pstate['version'] : '',
							'is_active'         => ! empty( $pstate['active'] ) ? 1 : 0,
						);
					}
				}
			}
		} else {
			$p_state = $this->get_plugin_state( $slug );
			$event_items[] = array(
				'product_type'      => 'plugin',
				'slug'              => $slug,
				'identifier'        => isset( $p_state['plugin_file'] ) ? (string) $p_state['plugin_file'] : '',
				'installed_version' => isset( $p_state['version'] ) ? (string) $p_state['version'] : '',
				'is_active'         => ! empty( $p_state['active'] ) ? 1 : 0,
			);
		}
		if ( ! empty( $event_items ) ) {
			\pluxix\Client\Reporter::send_event( 'install', array( 'items' => $event_items ) );
			\pluxix\Client\Reporter::send_heartbeat( true );
		}

		// Clear cached marketplace to refresh states.
		$this->clear_cache_internal();

		wp_send_json_success( array( 'result' => $result, 'bundle' => $bundle ) );
	}

	public function activate(): void {
		$this->verify();

		$slug = isset( $_POST['slug'] ) ? sanitize_key( wp_unslash( $_POST['slug'] ) ) : '';
		if ( '' === $slug ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid slug.', 'pluxix-client' ) ), 400 );
		}

		$plugin_file = $this->find_plugin_file_by_slug( $slug );
		if ( '' === $plugin_file ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Plugin is not installed.', 'pluxix-client' ) ), 400 );
		}

		$result = Package_Installer::activate_plugin_file( $plugin_file );
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ), 400 );
		}

		wp_send_json_success( array( 'result' => $result ) );
	}

	public function update(): void {
		$this->verify();

		$ob_level = ob_get_level();
		ob_start();

		$type = 'plugin';
		$slug = '';
		$new_version = '';
		$result = null;
		$prev_state = array();

		try {
			$type        = isset( $_POST['type'] ) ? sanitize_key( wp_unslash( $_POST['type'] ) ) : 'plugin';
			$slug        = isset( $_POST['slug'] ) ? sanitize_key( wp_unslash( $_POST['slug'] ) ) : '';
			$new_version = isset( $_POST['new_version'] ) ? sanitize_text_field( wp_unslash( $_POST['new_version'] ) ) : '';

			if ( '' === $slug ) {
				throw new \Exception( esc_html__( 'Invalid slug.', 'pluxix-client' ) );
			}

			$type = ( 'theme' === $type ) ? 'theme' : 'plugin';

			$download_url = Server_Api::get_download_url( $type, $slug );
			if ( '' === $download_url ) {
				throw new \Exception( esc_html__( 'Server URL is not configured.', 'pluxix-client' ) );
			}

			if ( 'theme' === $type ) {
				$state      = $this->get_theme_state( $slug );
				$prev_state = $state;
				$stylesheet = isset( $state['stylesheet'] ) ? (string) $state['stylesheet'] : '';
				$result     = Package_Installer::update_theme_from_package( $download_url, $stylesheet, $slug, $new_version ?: '0.0.0' );
			} else {
				$state       = $this->get_plugin_state( $slug );
				$prev_state  = $state;
				$plugin_file = isset( $state['plugin_file'] ) ? (string) $state['plugin_file'] : '';
				$result      = Package_Installer::update_plugin_from_package( $download_url, $plugin_file, $slug, $new_version ?: '0.0.0' );
			}
		} catch ( \Throwable $e ) {
			while ( ob_get_level() > $ob_level ) {
				ob_end_clean();
			}
			wp_send_json_error( array( 'message' => $e->getMessage() ), 500 );
		}

		// Clean any warnings/notices that may have been printed during upgrade.
		while ( ob_get_level() > $ob_level ) {
			ob_end_clean();
		}

		if ( is_wp_error( $result ) ) {
			wp_send_json_error( array( 'message' => $result->get_error_message() ), 400 );
		}

		// If upgrader returned updated identifiers, persist them.
		$map = $this->get_map();
		if ( 'theme' === $type ) {
			if ( is_array( $result ) && ! empty( $result['stylesheet'] ) ) {
				$map['themes'][ $slug ] = (string) $result['stylesheet'];
			}
		} else {
			if ( is_array( $result ) && ! empty( $result['plugin_file'] ) ) {
				$map['plugins'][ $slug ] = (string) $result['plugin_file'];
			}
		}

		$bundle = null;
		if ( 'theme' === $type ) {
			$accessRes = Server_Api::check_access();
			$access    = ( is_wp_error( $accessRes ) ) ? array() : ( isset( $accessRes['access'] ) ? $this->force_access_action_url( $accessRes['access'] ) : array() );
			$bundle    = $this->sync_theme_bundles( $slug, $access, $map );
		}

		$this->update_map( $map );
		$this->clear_cache_internal();

		// Telemetry: report updates.
		$new_state = ( 'theme' === $type ) ? $this->get_theme_state( $slug ) : $this->get_plugin_state( $slug );
		$from_ver = isset( $prev_state['version'] ) ? (string) $prev_state['version'] : '';
		$to_ver   = isset( $new_state['version'] ) ? (string) $new_state['version'] : '';
		$identifier = '';
		$is_active = 0;
		if ( 'theme' === $type ) {
			$identifier = isset( $new_state['stylesheet'] ) ? (string) $new_state['stylesheet'] : '';
			$is_active  = ! empty( $new_state['active'] ) ? 1 : 0;
		} else {
			$identifier = isset( $new_state['plugin_file'] ) ? (string) $new_state['plugin_file'] : '';
			$is_active  = ! empty( $new_state['active'] ) ? 1 : 0;
		}
		\pluxix\Client\Reporter::send_event( 'update', array(
			'items' => array(
				array(
					'product_type' => (string) $type,
					'slug'         => (string) $slug,
					'identifier'   => $identifier,
					'from_version' => $from_ver,
					'to_version'   => $to_ver,
					'installed_version' => $to_ver,
					'is_active'    => $is_active,
				),
			),
		) );
		\pluxix\Client\Reporter::send_heartbeat( true );

		wp_send_json_success( array( 'result' => $result, 'bundle' => $bundle ) );
	}

	public function get_updates(): void {
		$this->verify();

		$map = $this->get_map();

		$installed = array(
			'plugins' => array(),
			'themes'  => array(),
		);

		$updates = array(
			'plugins' => array(),
			'themes'  => array(),
		);

		// Only list items installed via pluxix (tracked in map).
		$plugin_slugs = ( isset( $map['plugins'] ) && is_array( $map['plugins'] ) ) ? array_keys( $map['plugins'] ) : array();
		foreach ( $plugin_slugs as $slug ) {
			$slug  = sanitize_key( (string) $slug );
			$state = $this->get_plugin_state( $slug );
			if ( ! $state['installed'] ) {
				continue;
			}

			$remote_item = $this->get_remote_item_cached( 'plugin', $slug );
			$remote_ver  = ( is_array( $remote_item ) && isset( $remote_item['version'] ) ) ? (string) $remote_item['version'] : '';
			$remote_vn   = $this->norm_ver( $remote_ver );
			$local_vn    = $this->norm_ver( (string) $state['version'] );
			$update_avail = ( '' !== $remote_vn && '' !== $local_vn ) ? version_compare( $remote_vn, $local_vn, '>' ) : false;

			$row = array(
				'type'            => 'plugin',
				'slug'            => $slug,
				'name'            => ( is_array( $remote_item ) && ! empty( $remote_item['name'] ) ) ? (string) $remote_item['name'] : $slug,
				'current'         => (string) $state['version'],
				'new'             => $remote_ver,
				'description'     => ( is_array( $remote_item ) && isset( $remote_item['description'] ) ) ? (string) $remote_item['description'] : '',
				'changelog'       => ( is_array( $remote_item ) && isset( $remote_item['changelog'] ) ) ? (string) $remote_item['changelog'] : '',
				'access_level'    => ( is_array( $remote_item ) && isset( $remote_item['access_level'] ) ) ? (string) $remote_item['access_level'] : 'free',
				'image_url'       => ( is_array( $remote_item ) && isset( $remote_item['image_url'] ) ) ? (string) $remote_item['image_url'] : '',
				'demo_url'        => ( is_array( $remote_item ) && isset( $remote_item['demo_url'] ) ) ? (string) $remote_item['demo_url'] : '',
				'update_available'=> (bool) $update_avail,
			);

			$installed['plugins'][] = $row;
			if ( $update_avail ) {
				$updates['plugins'][] = $row;
			}
		}

		$theme_slugs = ( isset( $map['themes'] ) && is_array( $map['themes'] ) ) ? array_keys( $map['themes'] ) : array();
		foreach ( $theme_slugs as $slug ) {
			$slug  = sanitize_key( (string) $slug );
			$state = $this->get_theme_state( $slug );
			if ( ! $state['installed'] ) {
				continue;
			}

			$remote_item = $this->get_remote_item_cached( 'theme', $slug );
			$remote_ver  = ( is_array( $remote_item ) && isset( $remote_item['version'] ) ) ? (string) $remote_item['version'] : '';
			$remote_vn   = $this->norm_ver( $remote_ver );
			$local_vn    = $this->norm_ver( (string) $state['version'] );
			$update_avail = ( '' !== $remote_vn && '' !== $local_vn ) ? version_compare( $remote_vn, $local_vn, '>' ) : false;

			$row = array(
				'type'            => 'theme',
				'slug'            => $slug,
				'name'            => ( is_array( $remote_item ) && ! empty( $remote_item['name'] ) ) ? (string) $remote_item['name'] : $slug,
				'current'         => (string) $state['version'],
				'new'             => $remote_ver,
				'description'     => ( is_array( $remote_item ) && isset( $remote_item['description'] ) ) ? (string) $remote_item['description'] : '',
				'changelog'       => ( is_array( $remote_item ) && isset( $remote_item['changelog'] ) ) ? (string) $remote_item['changelog'] : '',
				'access_level'    => ( is_array( $remote_item ) && isset( $remote_item['access_level'] ) ) ? (string) $remote_item['access_level'] : 'free',
				'image_url'       => ( is_array( $remote_item ) && isset( $remote_item['screenshot_url'] ) && $remote_item['screenshot_url'] ) ? (string) $remote_item['screenshot_url'] : ( ( is_array( $remote_item ) && isset( $remote_item['image_url'] ) ) ? (string) $remote_item['image_url'] : '' ),
				'screenshot_url'  => ( is_array( $remote_item ) && isset( $remote_item['screenshot_url'] ) ) ? (string) $remote_item['screenshot_url'] : '',
				'demo_url'        => ( is_array( $remote_item ) && isset( $remote_item['demo_url'] ) ) ? (string) $remote_item['demo_url'] : '',
				'update_available'=> (bool) $update_avail,
			);

			$installed['themes'][] = $row;
			if ( $update_avail ) {
				$updates['themes'][] = $row;
			}
		}

		wp_send_json_success(
			array(
				'installed' => $installed,
				'updates'   => $updates,
			)
		);
	}

	public function get_auto_update(): void {
		$this->verify();

		$enabled = (bool) get_option( 'pluxix_client_auto_update_enabled', false );

		$accessRes = \pluxix\Client\Api\Server_Api::check_access();
		$access    = ( ! is_wp_error( $accessRes ) && isset( $accessRes['access'] ) && is_array( $accessRes['access'] ) ) ? $this->force_access_action_url( $accessRes['access'] ) : array();

		wp_send_json_success(
			array(
				'enabled' => $enabled,
				'access'  => $access,
			)
		);
	}

	public function set_auto_update(): void {
		$this->verify();

		$enabled = ! empty( $_POST['enabled'] );

		$accessRes = \pluxix\Client\Api\Server_Api::check_access();
		if ( is_wp_error( $accessRes ) ) {
			wp_send_json_error( array( 'message' => $accessRes->get_error_message() ), 400 );
		}
		$access = isset( $accessRes['access'] ) && is_array( $accessRes['access'] ) ? $this->force_access_action_url( $accessRes['access'] ) : array();
		$level  = isset( $access['access_level'] ) ? (string) $access['access_level'] : 'free';
		if ( 'pro' !== $level ) {
			wp_send_json_error(
				array(
					'message'    => esc_html__( 'Auto Update is available for Pro subscribers only.', 'pluxix-client' ),
					'action_url' => $this->ready_licenses_url(),
				),
				403
			);
		}

		update_option( 'pluxix_client_auto_update_enabled', $enabled ? 1 : 0 );

		if ( $enabled ) {
			\pluxix\Client\Cron::ensure_scheduled();
		} else {
			\pluxix\Client\Cron::unschedule();
		}

		wp_send_json_success( array( 'enabled' => $enabled ) );
	}

	public function update_all(): void {
		$this->verify();

		$items = isset( $_POST['items'] ) ? (array) wp_unslash( $_POST['items'] ) : array();

		$results = array();
		$event_items = array();

		$accessRes = Server_Api::check_access();
		$access    = ( is_wp_error( $accessRes ) ) ? array() : ( isset( $accessRes['access'] ) ? $this->force_access_action_url( $accessRes['access'] ) : array() );

		foreach ( $items as $item ) {
			if ( ! is_array( $item ) || empty( $item['slug'] ) || empty( $item['type'] ) ) {
				continue;
			}
			$type = ( 'theme' === $item['type'] ) ? 'theme' : 'plugin';
			$slug = sanitize_key( (string) $item['slug'] );
			$new_version = isset( $item['new_version'] ) ? sanitize_text_field( (string) $item['new_version'] ) : '';

			$download_url = Server_Api::get_download_url( $type, $slug );
			if ( '' === $download_url ) {
				$results[] = array( 'slug' => $slug, 'type' => $type, 'success' => false, 'message' => esc_html__( 'Server URL is not configured.', 'pluxix-client' ) );
				continue;
			}

			if ( 'theme' === $type ) {
				$state = $this->get_theme_state( $slug );
				$stylesheet = isset( $state['stylesheet'] ) ? (string) $state['stylesheet'] : '';
				$res = Package_Installer::update_theme_from_package( $download_url, $stylesheet, $slug, $new_version ?: '0.0.0' );
			} else {
				$state = $this->get_plugin_state( $slug );
				$plugin_file = isset( $state['plugin_file'] ) ? (string) $state['plugin_file'] : '';
				$res = Package_Installer::update_plugin_from_package( $download_url, $plugin_file, $slug, $new_version ?: '0.0.0' );
			}

			if ( is_wp_error( $res ) ) {
				$results[] = array( 'slug' => $slug, 'type' => $type, 'success' => false, 'message' => $res->get_error_message() );
			} else {
				$from_ver = isset( $state['version'] ) ? (string) $state['version'] : '';
				$map = $this->get_map();

				if ( 'theme' === $type && is_array( $res ) && ! empty( $res['stylesheet'] ) ) {
					$map['themes'][ $slug ] = (string) $res['stylesheet'];
				}
				if ( 'plugin' === $type && is_array( $res ) && ! empty( $res['plugin_file'] ) ) {
					$map['plugins'][ $slug ] = (string) $res['plugin_file'];
				}

				$bundle = null;
				if ( 'theme' === $type ) {
					$bundle = $this->sync_theme_bundles( $slug, $access, $map );
				}

				$this->update_map( $map );

				$new_state = ( 'theme' === $type ) ? $this->get_theme_state( $slug ) : $this->get_plugin_state( $slug );
				$to_ver = isset( $new_state['version'] ) ? (string) $new_state['version'] : '';
				$identifier = '';
				$is_active = 0;
				if ( 'theme' === $type ) {
					$identifier = isset( $new_state['stylesheet'] ) ? (string) $new_state['stylesheet'] : '';
					$is_active  = ! empty( $new_state['active'] ) ? 1 : 0;
				} else {
					$identifier = isset( $new_state['plugin_file'] ) ? (string) $new_state['plugin_file'] : '';
					$is_active  = ! empty( $new_state['active'] ) ? 1 : 0;
				}
				$event_items[] = array(
					'product_type'      => (string) $type,
					'slug'              => (string) $slug,
					'identifier'        => $identifier,
					'from_version'      => $from_ver,
					'to_version'        => $to_ver,
					'installed_version' => $to_ver,
					'is_active'         => $is_active,
				);

				$results[] = array( 'slug' => $slug, 'type' => $type, 'success' => true, 'bundle' => $bundle );
			}
		}

		$this->clear_cache_internal();

		if ( ! empty( $event_items ) ) {
			\pluxix\Client\Reporter::send_event( 'update', array( 'bulk' => 1, 'items' => $event_items ) );
			\pluxix\Client\Reporter::send_heartbeat( true );
		}

		wp_send_json_success( array( 'results' => $results ) );
	}

	private function clear_cache_internal(): void {
		$keys = get_option( 'pluxix_client_cache_keys', array() );
		if ( is_array( $keys ) ) {
			foreach ( $keys as $k ) {
				delete_transient( (string) $k );
			}
		}
		update_option( 'pluxix_client_cache_keys', array() );
		delete_transient( 'rs_client_access' );
	}

	public function clear_cache(): void {
		$this->verify();
		$this->clear_cache_internal();
		wp_send_json_success();
	}

	public function sync(): void {
		$this->verify();
		$this->clear_cache_internal();
		// Warm access cache.
		$res = Server_Api::check_access();
		if ( ! is_wp_error( $res ) && isset( $res['access'] ) ) {
			set_transient( 'rs_client_access', $res['access'], Server_Api::get_cache_ttl() );
			update_option( 'pluxix_client_last_access', array( 'time' => time(), 'access' => $res['access'] ) );
		}
		\pluxix\Client\Reporter::send_heartbeat( true );
		wp_send_json_success();
	}
}
