<?php

namespace pluxix\Client\Api;

use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Server_Api {
	public static function get_server_url(): string {
		$encoded = defined( 'pluxix_CLIENT_SERVER_URL_B64' ) ? (string) pluxix_CLIENT_SERVER_URL_B64 : '';
		$decoded = '';
		if ( '' !== $encoded ) {
			$decoded = (string) base64_decode( $encoded, true );
		}

		$url = trim( $decoded );
		if ( '' === $url ) {
			return '';
		}

		// Allow overriding in development environments.
		$url = (string) apply_filters( 'pluxix_client_server_url', $url );
		$url = untrailingslashit( trim( $url ) );

		return $url;
	}

	public static function get_account_email(): string {
		// Settings page is removed; default to site admin email.
		$email = (string) get_option( 'admin_email', '' );
		return sanitize_email( $email );
	}

	public static function get_cache_ttl(): int {
		$ttl = 300;
		return max( 60, min( 86400, $ttl ) );
	}

	public static function get_site_domain(): string {
		$domain = wp_parse_url( home_url(), PHP_URL_HOST );
		return $domain ? (string) $domain : '';
	}

	public static function get_site_ip(): string {
		$ip = isset( $_SERVER['SERVER_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_ADDR'] ) ) : '';
		return $ip;
	}

	private static function api_base(): string {
		$server_url = self::get_server_url();
		return $server_url ? $server_url . '/wp-json/pluxix-store/v1' : '';
	}

	public static function request( string $method, string $endpoint, array $params = array() ) {
		$base = self::api_base();
		if ( '' === $base ) {
			return new WP_Error( 'pluxix_client_no_server', esc_html__( 'Server URL is not configured.', 'pluxix-client' ) );
		}

		$url = $base . $endpoint;

		$args = array(
			'timeout' => 25,
			'headers' => array(
				'Accept'        => 'application/json',
				'Cache-Control' => 'no-cache',
				'Pragma'        => 'no-cache',
			),
			'method'  => strtoupper( $method ),
		);

		if ( 'GET' === strtoupper( $method ) ) {
			if ( ! empty( $params ) ) {
				$url = add_query_arg( $params, $url );
			}
		} else {
			$args['headers']['Content-Type'] = 'application/json';
			$args['body']                    = wp_json_encode( $params );
		}

		$response = wp_remote_request( $url, $args );
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$code = (int) wp_remote_retrieve_response_code( $response );
		$body = (string) wp_remote_retrieve_body( $response );

		$data = json_decode( $body, true );

		if ( $code < 200 || $code >= 300 ) {
			$message = is_array( $data ) && isset( $data['message'] ) ? (string) $data['message'] : esc_html__( 'Server request failed.', 'pluxix-client' );
			$err     = new WP_Error( 'pluxix_client_server_error', $message, array( 'status' => $code, 'raw' => $data ) );
			return $err;
		}

		if ( null === $data ) {
			return new WP_Error( 'pluxix_client_invalid_json', esc_html__( 'Invalid server response.', 'pluxix-client' ) );
		}

		return $data;
	}

	public static function check_access() {
		return self::request(
			'POST',
			'/access',
			array(
				'domain'        => self::get_site_domain(),
				'account_email' => self::get_account_email(),
				'site_ip'       => self::get_site_ip(),
			)
		);
	}

	/**
	 * Telemetry: heartbeat (site + installed items).
	 */
	public static function send_heartbeat( array $payload ) {
		return self::request( 'POST', '/client/heartbeat', $payload );
	}

	/**
	 * Telemetry: event log (install/update/remove).
	 */
	public static function send_event( array $payload ) {
		return self::request( 'POST', '/client/event', $payload );
	}

	public static function get_marketplace( string $type, string $search = '', int $category = 0, int $page = 1, int $per_page = 0 ) {
		$endpoint = ( 'theme' === $type ) ? '/themes' : '/plugins';

		$params = array(
			'domain'        => self::get_site_domain(),
			'account_email' => self::get_account_email(),
		);

		if ( '' !== $search ) {
			$params['s'] = $search;
		}

		if ( $category ) {
			$params['category'] = $category;
		}

		if ( $per_page > 0 ) {
			$params['per_page'] = max( 1, min( 100, (int) $per_page ) );
			$params['page']     = max( 1, (int) $page );
		}

		// Add cache-buster to reduce stale results when server-side caches/CDNs are involved.
		$params['rs_ts'] = time();

		return self::request( 'GET', $endpoint, $params );
	}

	public static function get_item_info( string $type, string $slug ) {
		$slug = sanitize_key( (string) $slug );
		if ( '' === $slug ) {
			return new WP_Error( 'pluxix_client_invalid_slug', esc_html__( 'Invalid slug.', 'pluxix-client' ) );
		}

		$endpoint = ( 'theme' === $type ) ? '/theme-info/' . rawurlencode( $slug ) : '/plugin-info/' . rawurlencode( $slug );

		$params = array(
			'domain'        => self::get_site_domain(),
			'account_email' => self::get_account_email(),
			'rs_ts'         => time(),
		);

		return self::request( 'GET', $endpoint, $params );
	}

	public static function get_download_url( string $type, string $slug ): string {
		$base = self::api_base();
		if ( '' === $base ) {
			return '';
		}

		$endpoint = ( 'theme' === $type ) ? '/download/theme/' . rawurlencode( $slug ) : '/download/plugin/' . rawurlencode( $slug );

		$url = $base . $endpoint;

		return add_query_arg(
			array(
				'domain'        => self::get_site_domain(),
				'account_email' => self::get_account_email(),
				'site_ip'       => self::get_site_ip(),
			),
			$url
		);
	}

	public static function get_theme_child_download_url( string $slug ): string {
		$base = self::api_base();
		if ( '' === $base ) {
			return '';
		}
		$slug = sanitize_key( $slug );
		$endpoint = '/download/theme-child/' . rawurlencode( $slug );
		$url = $base . $endpoint;
		return add_query_arg(
			array(
				'domain'        => self::get_site_domain(),
				'account_email' => self::get_account_email(),
				'site_ip'       => self::get_site_ip(),
			),
			$url
		);
	}
}
