<?php

namespace pluxix\Client;

use pluxix\Client\Api\Server_Api;
use pluxix\Client\Services\Package_Installer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Cron {
	private static $instance = null;

	public const HOOK = 'pluxix_client_weekly_auto_update';

	public static function instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	private function __construct() {
		add_filter( 'cron_schedules', array( $this, 'add_weekly_schedule' ) );
		add_action( self::HOOK, array( $this, 'run_weekly_auto_update' ) );
	}

	public function add_weekly_schedule( array $schedules ): array {
		if ( ! isset( $schedules['weekly'] ) ) {
			$schedules['weekly'] = array(
				'interval' => 7 * DAY_IN_SECONDS,
				'display'  => esc_html__( 'Once Weekly', 'pluxix-client' ),
			);
		}
		return $schedules;
	}

	public static function ensure_scheduled(): void {
		if ( ! wp_next_scheduled( self::HOOK ) ) {
			// Add a small delay so scheduling works even during activation.
			wp_schedule_event( time() + HOUR_IN_SECONDS, 'weekly', self::HOOK );
		}
	}

	public static function unschedule(): void {
		$ts = wp_next_scheduled( self::HOOK );
		while ( $ts ) {
			wp_unschedule_event( $ts, self::HOOK );
			$ts = wp_next_scheduled( self::HOOK );
		}
	}

	private function get_map(): array {
		$map = get_option( 'pluxix_client_map', array() );
		$map = is_array( $map ) ? $map : array();
		return wp_parse_args(
			$map,
			array(
				'plugins' => array(),
				'themes'  => array(),
				'theme_children' => array(),
			)
		);
	}

	private function update_map( array $map ): void {
		update_option( 'pluxix_client_map', $map );
	}

	private function norm_ver( string $v ): string {
		$v = trim( $v );
		$v = preg_replace( '/^v/i', '', $v );
		$v = preg_replace( '/[^0-9.]/', '', $v );
		return is_string( $v ) ? $v : '';
	}

	private function get_plugin_version( string $plugin_file ): string {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		$plugins = get_plugins();
		return isset( $plugins[ $plugin_file ]['Version'] ) ? (string) $plugins[ $plugin_file ]['Version'] : '';
	}

	public function run_weekly_auto_update(): void {
		$enabled = (bool) get_option( 'pluxix_client_auto_update_enabled', false );
		if ( ! $enabled ) {
			return;
		}

		$accessRes = Server_Api::check_access();
		if ( is_wp_error( $accessRes ) ) {
			return;
		}
		$access = isset( $accessRes['access'] ) && is_array( $accessRes['access'] ) ? $accessRes['access'] : array();
		$level  = isset( $access['access_level'] ) ? (string) $access['access_level'] : 'free';
		if ( 'pro' !== $level ) {
			// Safety: auto update should only run for Pro.
			return;
		}

		$map = $this->get_map();
		$event_items = array();

		// Update plugins tracked by pluxix.
		if ( isset( $map['plugins'] ) && is_array( $map['plugins'] ) ) {
			foreach ( array_keys( $map['plugins'] ) as $slug ) {
				$slug = sanitize_key( (string) $slug );
				$plugin_file = isset( $map['plugins'][ $slug ] ) ? (string) $map['plugins'][ $slug ] : '';
				if ( '' === $plugin_file || ! file_exists( WP_PLUGIN_DIR . '/' . $plugin_file ) ) {
					continue;
				}

				$local_ver   = $this->get_plugin_version( $plugin_file );
				$local_norm  = $this->norm_ver( $local_ver );
				$itemRes     = Server_Api::get_item_info( 'plugin', $slug );
				if ( is_wp_error( $itemRes ) ) {
					continue;
				}
				$item = isset( $itemRes['item'] ) && is_array( $itemRes['item'] ) ? $itemRes['item'] : array();
				$remote_ver  = isset( $item['version'] ) ? (string) $item['version'] : '';
				$remote_norm = $this->norm_ver( $remote_ver );
				if ( '' === $remote_norm || '' === $local_norm || ! version_compare( $remote_norm, $local_norm, '>' ) ) {
					continue;
				}

				$download_url = Server_Api::get_download_url( 'plugin', $slug );
				if ( '' === $download_url ) {
					continue;
				}

				$res = Package_Installer::update_plugin_from_package( $download_url, $plugin_file, $slug, $remote_ver ?: '0.0.0' );
				if ( is_array( $res ) && ! empty( $res['plugin_file'] ) && (string) $res['plugin_file'] !== $plugin_file ) {
					$map['plugins'][ $slug ] = (string) $res['plugin_file'];
				}
				if ( ! is_wp_error( $res ) ) {
					$event_items[] = array(
						'product_type' => 'plugin',
						'slug' => (string) $slug,
						'from_version' => $local_ver,
						'to_version' => $remote_ver,
						'installed_version' => $remote_ver,
						'identifier' => ( is_array( $res ) && ! empty( $res['plugin_file'] ) ) ? (string) $res['plugin_file'] : $plugin_file,
						'is_active' => is_plugin_active( $plugin_file ) ? 1 : 0,
					);
				}
			}
		}

		// Update themes tracked by pluxix.
		if ( isset( $map['themes'] ) && is_array( $map['themes'] ) ) {
			foreach ( array_keys( $map['themes'] ) as $slug ) {
				$slug = sanitize_key( (string) $slug );
				$stylesheet = isset( $map['themes'][ $slug ] ) ? (string) $map['themes'][ $slug ] : '';
				if ( '' === $stylesheet ) {
					continue;
				}
				$theme = wp_get_theme( $stylesheet );
				if ( ! $theme || ! $theme->exists() ) {
					continue;
				}
				$local_ver   = (string) $theme->get( 'Version' );
				$local_norm  = $this->norm_ver( $local_ver );

				$itemRes     = Server_Api::get_item_info( 'theme', $slug );
				if ( is_wp_error( $itemRes ) ) {
					continue;
				}
				$item = isset( $itemRes['item'] ) && is_array( $itemRes['item'] ) ? $itemRes['item'] : array();
				$remote_ver  = isset( $item['version'] ) ? (string) $item['version'] : '';
				$remote_norm = $this->norm_ver( $remote_ver );
				if ( '' === $remote_norm || '' === $local_norm || ! version_compare( $remote_norm, $local_norm, '>' ) ) {
					continue;
				}

				$download_url = Server_Api::get_download_url( 'theme', $slug );
				if ( '' === $download_url ) {
					continue;
				}

				$res = Package_Installer::update_theme_from_package( $download_url, $stylesheet, $slug, $remote_ver ?: '0.0.0' );
				if ( is_array( $res ) && ! empty( $res['stylesheet'] ) && (string) $res['stylesheet'] !== $stylesheet ) {
					$map['themes'][ $slug ] = (string) $res['stylesheet'];
				}
				if ( ! is_wp_error( $res ) ) {
					$event_items[] = array(
						'product_type' => 'theme',
						'slug' => (string) $slug,
						'from_version' => $local_ver,
						'to_version' => $remote_ver,
						'installed_version' => $remote_ver,
						'identifier' => ( is_array( $res ) && ! empty( $res['stylesheet'] ) ) ? (string) $res['stylesheet'] : $stylesheet,
						'is_active' => ( ( wp_get_theme() instanceof \WP_Theme ) && ( ( is_array( $res ) && ! empty( $res['stylesheet'] ) ? (string) $res['stylesheet'] : $stylesheet ) === wp_get_theme()->get_stylesheet() ) ) ? 1 : 0,
					);
				}
			}
		}

		$this->update_map( $map );
		update_option( 'pluxix_client_last_auto_update', time() );

		if ( ! empty( $event_items ) ) {
			Reporter::send_event( 'auto_update', array( 'items' => $event_items ) );
			Reporter::send_heartbeat( true );
		}
	}
}
