<?php

namespace pluxix\Client;

use pluxix\Client\Api\Server_Api;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Sends lightweight telemetry to the PLUXIX Server for installation monitoring.
 */
class Reporter {
	public const HOOK_DAILY_HEARTBEAT = 'pluxix_client_daily_heartbeat';
	private const OPT_LAST_HEARTBEAT = 'pluxix_client_last_heartbeat';

	public static function init(): void {
		add_action( self::HOOK_DAILY_HEARTBEAT, array( self::class, 'send_heartbeat' ) );
		add_action( 'init', array( self::class, 'ensure_scheduled' ) );
	}

	/**
	 * Ensure the daily heartbeat cron is scheduled (useful on plugin upgrades where
	 * activation hooks do not run).
	 */
	public static function ensure_scheduled(): void {
		if ( wp_next_scheduled( self::HOOK_DAILY_HEARTBEAT ) ) {
			return;
		}
		self::schedule();
		// If we've never sent a heartbeat, do one now so the server can show the site immediately.
		if ( (int) get_option( self::OPT_LAST_HEARTBEAT, 0 ) <= 0 ) {
			self::send_heartbeat( true );
		}
	}

	public static function activate(): void {
		self::schedule();
		// Send an initial heartbeat.
		self::send_heartbeat( true );
		self::send_event( 'activate', array( 'client_version' => defined( 'pluxix_CLIENT_VERSION' ) ? pluxix_CLIENT_VERSION : '' ) );
	}

	public static function deactivate(): void {
		self::unschedule();
		self::send_event( 'deactivate', array( 'client_version' => defined( 'pluxix_CLIENT_VERSION' ) ? pluxix_CLIENT_VERSION : '' ) );
	}

	public static function schedule(): void {
		if ( ! wp_next_scheduled( self::HOOK_DAILY_HEARTBEAT ) ) {
			wp_schedule_event( time() + HOUR_IN_SECONDS, 'daily', self::HOOK_DAILY_HEARTBEAT );
		}
	}

	public static function unschedule(): void {
		$ts = wp_next_scheduled( self::HOOK_DAILY_HEARTBEAT );
		while ( $ts ) {
			wp_unschedule_event( $ts, self::HOOK_DAILY_HEARTBEAT );
			$ts = wp_next_scheduled( self::HOOK_DAILY_HEARTBEAT );
		}
	}

	/**
	 * Send a heartbeat (site + installed PLUXIX items). If $force is false, this will be rate-limited.
	 */
	public static function send_heartbeat( bool $force = false ): void {
		$last = (int) get_option( self::OPT_LAST_HEARTBEAT, 0 );
		if ( ! $force && $last > 0 && ( time() - $last ) < 6 * HOUR_IN_SECONDS ) {
			return;
		}

		$payload = self::build_payload();
		Server_Api::send_heartbeat( $payload );
		update_option( self::OPT_LAST_HEARTBEAT, time(), false );
	}

	/**
	 * Send a single event.
	 */
	public static function send_event( string $event_type, array $payload = array() ): void {
		$event_type = sanitize_key( $event_type );
		if ( '' === $event_type ) {
			return;
		}
		// Attach actor context automatically so the server can show "who" performed the action.
		if ( ! isset( $payload['actor'] ) ) {
			$payload['actor'] = self::build_actor();
		}
		$base = self::build_base();
		$body = array_merge(
			$base,
			array(
				'event_type' => $event_type,
				'payload'    => $payload,
			)
		);
		Server_Api::send_event( $body );
	}

	private static function build_actor(): array {
		if ( function_exists( 'is_user_logged_in' ) && is_user_logged_in() ) {
			$user = wp_get_current_user();
			if ( $user instanceof \WP_User && $user->ID ) {
				return array(
					'type'  => 'user',
					'id'    => (int) $user->ID,
					'login' => (string) $user->user_login,
					'email' => (string) $user->user_email,
				);
			}
		}
		return array(
			'type' => 'system',
			'name' => ( function_exists( 'wp_doing_cron' ) && wp_doing_cron() ) ? 'cron' : 'system',
		);
	}

	private static function build_base(): array {
		$domain = wp_parse_url( home_url(), PHP_URL_HOST );
		return array(
			'domain'        => is_string( $domain ) ? strtolower( $domain ) : '',
			'site_url'      => home_url(),
			'admin_email'   => get_option( 'admin_email' ),
			'client_version'=> defined( 'pluxix_CLIENT_VERSION' ) ? pluxix_CLIENT_VERSION : '',
			'wp_version'    => get_bloginfo( 'version' ),
			'php_version'   => PHP_VERSION,
			'locale'        => function_exists( 'determine_locale' ) ? determine_locale() : get_locale(),
			'timezone'      => function_exists( 'wp_timezone_string' ) ? wp_timezone_string() : '',
			'multisite'     => is_multisite() ? 1 : 0,
		);
	}

	private static function build_payload(): array {
		$base = self::build_base();
		$base['installed'] = self::get_installed_items();
		$base['counts']    = array(
			'plugins' => count( get_plugins() ),
			'themes'  => count( wp_get_themes() ),
		);
		return $base;
	}

	/**
	 * Returns items installed via PLUXIX (from the stored map + current WP state).
	 */
	private static function get_installed_items(): array {
		$items = array();

		$map = get_option( 'pluxix_client_map', array() );
		if ( ! is_array( $map ) ) {
			$map = array();
		}
		$plugins_map = isset( $map['plugins'] ) && is_array( $map['plugins'] ) ? $map['plugins'] : array();
		$themes_map  = isset( $map['themes'] ) && is_array( $map['themes'] ) ? $map['themes'] : array();
		$children_map = isset( $map['theme_children'] ) && is_array( $map['theme_children'] ) ? $map['theme_children'] : array();

		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		$all_plugins = get_plugins();

		foreach ( $plugins_map as $slug => $plugin_file ) {
			$slug = sanitize_title( (string) $slug );
			$plugin_file = (string) $plugin_file;
			if ( '' === $slug ) {
				continue;
			}
			// Older maps might store only the slug. Try to infer the plugin file.
			if ( '' === $plugin_file || false === strpos( $plugin_file, '/' ) ) {
				$plugin_file = '';
				foreach ( $all_plugins as $file => $data ) {
					if ( 0 === strpos( $file, $slug . '/' ) ) {
						$plugin_file = (string) $file;
						break;
					}
				}
			}
			if ( '' === $plugin_file ) {
				continue;
			}
			$ver = '';
			$active = is_plugin_active( $plugin_file ) ? 1 : 0;
			if ( file_exists( WP_PLUGIN_DIR . '/' . $plugin_file ) ) {
				$info = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin_file, false, false );
				$ver  = isset( $info['Version'] ) ? (string) $info['Version'] : '';
			}
			$items[] = array(
				'product_type'      => 'plugin',
				'slug'              => $slug,
				'identifier'        => $plugin_file,
				'installed_version' => $ver,
				'is_active'         => $active,
			);
		}

		$current  = wp_get_theme();
		$current_stylesheet = $current instanceof \WP_Theme ? $current->get_stylesheet() : '';

		foreach ( $themes_map as $slug => $stylesheet ) {
			$slug = sanitize_title( (string) $slug );
			$stylesheet = (string) $stylesheet;
			if ( '' === $slug || '' === $stylesheet ) {
				continue;
			}
			$theme = wp_get_theme( $stylesheet );
			$ver = $theme instanceof \WP_Theme ? $theme->get( 'Version' ) : '';
			$items[] = array(
				'product_type'      => 'theme',
				'slug'              => $slug,
				'identifier'        => $stylesheet,
				'installed_version' => (string) $ver,
				'is_active'         => ( $stylesheet === $current_stylesheet ) ? 1 : 0,
			);
		}

		foreach ( $children_map as $slug => $stylesheet ) {
			$slug = sanitize_title( (string) $slug );
			$stylesheet = (string) $stylesheet;
			if ( '' === $slug || '' === $stylesheet ) {
				continue;
			}
			$theme = wp_get_theme( $stylesheet );
			$ver = $theme instanceof \WP_Theme ? $theme->get( 'Version' ) : '';
			$items[] = array(
				'product_type'      => 'theme_child',
				'slug'              => $slug,
				'identifier'        => $stylesheet,
				'installed_version' => (string) $ver,
				'is_active'         => ( $stylesheet === $current_stylesheet ) ? 1 : 0,
			);
		}

		return $items;
	}
}
