<?php

namespace pluxix\Client\Services;

use WP_Error;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Package_Installer {
	/**
	 * Recursively delete a directory using WP_Filesystem when possible.
	 */
	private static function delete_dir( string $path ): void {
		if ( empty( $path ) ) {
			return;
		}
		$path = untrailingslashit( $path );
		global $wp_filesystem;
		if ( isset( $wp_filesystem ) && $wp_filesystem ) {
			// phpcs:ignore WordPressVIPMinimum.Functions.RestrictedFunctions.file_ops_rmdir
			$wp_filesystem->delete( $path, true );
			return;
		}

		// Fallback: best-effort using PHP.
		if ( ! is_dir( $path ) ) {
			return;
		}
		$it = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $path, \FilesystemIterator::SKIP_DOTS ),
			\RecursiveIteratorIterator::CHILD_FIRST
		);
		foreach ( $it as $file ) {
			/** @var \SplFileInfo $file */
			if ( $file->isDir() ) {
				@rmdir( $file->getRealPath() );
			} else {
				@unlink( $file->getRealPath() );
			}
		}
		@rmdir( $path );
	}

	/**
	 * Find the directory within an extracted package that contains a specific file.
	 *
	 * @return string Absolute path to the directory containing the file, or empty.
	 */
	private static function find_dir_containing_file( string $base_dir, string $filename, int $max_depth = 4 ): string {
		$base_dir = untrailingslashit( $base_dir );
		if ( ! is_dir( $base_dir ) ) {
			return '';
		}

		$rii = new \RecursiveIteratorIterator(
			new \RecursiveDirectoryIterator( $base_dir, \FilesystemIterator::SKIP_DOTS ),
			\RecursiveIteratorIterator::SELF_FIRST
		);
		foreach ( $rii as $file ) {
			/** @var \SplFileInfo $file */
			if ( $rii->getDepth() > $max_depth ) {
				continue;
			}
			if ( $file->isFile() && strtolower( $file->getFilename() ) === strtolower( $filename ) ) {
				$dir = dirname( $file->getRealPath() );
				return is_dir( $dir ) ? $dir : '';
			}
		}
		return '';
	}

	/**
	 * If the extracted directory contains a single top-level folder, return it.
	 */
	private static function maybe_single_root( string $base_dir ): string {
		$base_dir = untrailingslashit( $base_dir );
		if ( ! is_dir( $base_dir ) ) {
			return $base_dir;
		}
		$entries = array_values(
			array_filter(
				scandir( $base_dir ) ?: array(),
				static function ( $n ) {
					return '.' !== $n && '..' !== $n;
				}
			)
		);
		if ( 1 === count( $entries ) ) {
			$maybe = $base_dir . '/' . $entries[0];
			if ( is_dir( $maybe ) ) {
				return $maybe;
			}
		}
		return $base_dir;
	}

	/**
	 * Normalize version strings for reliable comparisons.
	 */
	private static function norm_ver( string $v ): string {
		$v = trim( $v );
		$v = preg_replace( '/^v/i', '', $v );
		$v = preg_replace( '/[^0-9.]/', '', $v );
		return is_string( $v ) ? $v : '';
	}
	private static function ensure_filesystem() {
		require_once ABSPATH . 'wp-admin/includes/file.php';

		if ( WP_Filesystem() ) {
			return true;
		}

		return new WP_Error(
			'pluxix_client_filesystem',
			esc_html__( 'Filesystem access is required to install or update packages on this site.', 'pluxix-client' )
		);
	}

	private static function get_skin() {
		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

		return new class() extends \WP_Upgrader_Skin {
			public function header() {}
			public function footer() {}
			public function feedback( $string, ...$args ) {}
		};
	}

	public static function install_plugin( string $download_url ) {
		$fs = self::ensure_filesystem();
		if ( is_wp_error( $fs ) ) {
			return $fs;
		}

		require_once ABSPATH . 'wp-admin/includes/plugin.php';
		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

		$upgrader = new \Plugin_Upgrader( self::get_skin() );

		$result = $upgrader->install( $download_url );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		if ( ! $result ) {
			return new WP_Error( 'pluxix_client_install_failed', esc_html__( 'Plugin installation failed.', 'pluxix-client' ) );
		}

		$plugin_file = method_exists( $upgrader, 'plugin_info' ) ? (string) $upgrader->plugin_info() : '';

		return array(
			'success'     => true,
			'plugin_file' => $plugin_file,
		);
	}

	public static function update_plugin( string $download_url ) {
		return new WP_Error( 'pluxix_client_bad_call', esc_html__( 'Invalid update call.', 'pluxix-client' ) );
	}

	public static function update_plugin_from_package( string $download_url, string $plugin_file, string $slug, string $new_version ) {
		$fs = self::ensure_filesystem();
		if ( is_wp_error( $fs ) ) {
			return $fs;
		}

		// Reduce the chance of timeouts on slower hosting environments.
		@set_time_limit( 0 );
		if ( function_exists( 'wp_raise_memory_limit' ) ) {
			wp_raise_memory_limit( 'admin' );
		}

		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		if ( '' === $plugin_file ) {
			return new WP_Error( 'pluxix_client_missing_plugin', esc_html__( 'Plugin is not installed.', 'pluxix-client' ) );
		}

		$plugins_before = get_plugins();
		$before_ver     = isset( $plugins_before[ $plugin_file ]['Version'] ) ? (string) $plugins_before[ $plugin_file ]['Version'] : '';
		$before_ver_n   = self::norm_ver( $before_ver );
		$new_ver_n      = self::norm_ver( (string) $new_version );

		$plugin_dir = dirname( $plugin_file );
		$plugin_dir = ( '.' === $plugin_dir ) ? $slug : $plugin_dir;
		$plugin_dir = sanitize_key( $plugin_dir );

		// Download the package.
		$tmp_file = download_url( $download_url, 300 );
		if ( is_wp_error( $tmp_file ) ) {
			return $tmp_file;
		}

		$working_dir = trailingslashit( WP_CONTENT_DIR ) . 'upgrade/pluxix-client-' . time() . '-' . wp_generate_password( 6, false, false );
		wp_mkdir_p( $working_dir );

		$unzipped = unzip_file( $tmp_file, $working_dir );
		@unlink( $tmp_file );
		if ( is_wp_error( $unzipped ) ) {
			self::delete_dir( $working_dir );
			return $unzipped;
		}

		$base = self::maybe_single_root( $working_dir );
		$main = basename( $plugin_file );
		$src  = self::find_dir_containing_file( $base, $main, 6 );
		if ( '' === $src ) {
			$src = $base;
		}

		// Detect version inside the package (helps diagnose version mismatch between metadata and zip).
		$package_ver = '';
		if ( file_exists( trailingslashit( $src ) . $main ) ) {
			$package_data = get_plugin_data( trailingslashit( $src ) . $main, false, false );
			if ( is_array( $package_data ) && isset( $package_data['Version'] ) ) {
				$package_ver = (string) $package_data['Version'];
			}
		}

		$dest = trailingslashit( WP_PLUGIN_DIR ) . $plugin_dir;
		self::delete_dir( $dest );
		wp_mkdir_p( $dest );

		$copy = copy_dir( $src, $dest );
		// Cleanup working dir.
		self::delete_dir( $working_dir );

		if ( is_wp_error( $copy ) ) {
			return $copy;
		}

		wp_clean_plugins_cache( true );

		$plugins_after = get_plugins();
		$after_ver     = isset( $plugins_after[ $plugin_file ]['Version'] ) ? (string) $plugins_after[ $plugin_file ]['Version'] : '';
		$after_ver_n   = self::norm_ver( $after_ver );

		// Verify version actually changed when a higher remote version was requested.
		if ( '' !== $new_ver_n && '' !== $before_ver_n && version_compare( $new_ver_n, $before_ver_n, '>' ) ) {
			if ( '' === $after_ver_n || version_compare( $after_ver_n, $new_ver_n, '<' ) ) {
				$extra = '';
				if ( $package_ver ) {
					$extra = ' | Package: ' . $package_ver;
				}
				return new WP_Error(
					'pluxix_client_update_verify_failed',
					sprintf(
						/* translators: 1: old version, 2: expected new version, 3: detected version, 4: package version */
						esc_html__( 'Update completed but version did not change. Old: %1$s | Expected: %2$s | Detected: %3$s%4$s', 'pluxix-client' ),
						$before_ver ?: '-',
						$new_version ?: '-',
						$after_ver ?: '-',
						$extra
					)
				);
			}
		}

		return array(
			'success'      => true,
			'version'      => $after_ver,
			'plugin_file'  => $plugin_file,
			'package_ver'  => $package_ver,
		);
	}

	public static function activate_plugin_file( string $plugin_file ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';

		if ( ! file_exists( WP_PLUGIN_DIR . '/' . $plugin_file ) ) {
			return new WP_Error( 'pluxix_client_missing_plugin', esc_html__( 'Plugin files not found.', 'pluxix-client' ) );
		}

		$result = activate_plugin( $plugin_file );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return array( 'success' => true );
	}

	public static function install_theme( string $download_url ) {
		$fs = self::ensure_filesystem();
		if ( is_wp_error( $fs ) ) {
			return $fs;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

		$upgrader = new \Theme_Upgrader( self::get_skin() );

		$result = $upgrader->install( $download_url );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		if ( ! $result ) {
			return new WP_Error( 'pluxix_client_install_failed', esc_html__( 'Theme installation failed.', 'pluxix-client' ) );
		}

		$stylesheet = '';
		if ( method_exists( $upgrader, 'theme_info' ) ) {
			$theme = $upgrader->theme_info();
			if ( $theme instanceof \WP_Theme ) {
				$stylesheet = $theme->get_stylesheet();
			}
		}

		if ( ! $stylesheet && is_array( $upgrader->result ) && isset( $upgrader->result['destination_name'] ) ) {
			$stylesheet = (string) $upgrader->result['destination_name'];
		}

		return array(
			'success'    => true,
			'stylesheet' => $stylesheet,
		);
	}

	public static function update_theme( string $download_url ) {
		$fs = self::ensure_filesystem();
		if ( is_wp_error( $fs ) ) {
			return $fs;
		}

		require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

		return new WP_Error( 'pluxix_client_bad_call', esc_html__( 'Invalid update call.', 'pluxix-client' ) );
	}

	public static function update_theme_from_package( string $download_url, string $stylesheet, string $slug, string $new_version ) {
		$fs = self::ensure_filesystem();
		if ( is_wp_error( $fs ) ) {
			return $fs;
		}

		// Reduce the chance of timeouts on slower hosting environments.
		@set_time_limit( 0 );
		if ( function_exists( 'wp_raise_memory_limit' ) ) {
			wp_raise_memory_limit( 'admin' );
		}

		require_once ABSPATH . 'wp-admin/includes/file.php';

		if ( '' === $stylesheet ) {
			return new WP_Error( 'pluxix_client_missing_theme', esc_html__( 'Theme is not installed.', 'pluxix-client' ) );
		}

		$before_ver   = (string) wp_get_theme( $stylesheet )->get( 'Version' );
		$before_ver_n = self::norm_ver( $before_ver );
		$new_ver_n    = self::norm_ver( (string) $new_version );

		$theme_root = get_theme_root( $stylesheet );
		if ( ! $theme_root ) {
			$theme_root = get_theme_root();
		}

		// Download the package.
		$tmp_file = download_url( $download_url, 300 );
		if ( is_wp_error( $tmp_file ) ) {
			return $tmp_file;
		}

		$working_dir = trailingslashit( WP_CONTENT_DIR ) . 'upgrade/pluxix-client-' . time() . '-' . wp_generate_password( 6, false, false );
		wp_mkdir_p( $working_dir );

		$unzipped = unzip_file( $tmp_file, $working_dir );
		@unlink( $tmp_file );
		if ( is_wp_error( $unzipped ) ) {
			self::delete_dir( $working_dir );
			return $unzipped;
		}

		$base = self::maybe_single_root( $working_dir );
		$src  = self::find_dir_containing_file( $base, 'style.css', 6 );
		if ( '' === $src ) {
			$src = $base;
		}

		// Detect version inside the package (helps diagnose version mismatch between metadata and zip).
		$package_ver = '';
		$style_path  = trailingslashit( $src ) . 'style.css';
		if ( file_exists( $style_path ) ) {
			$data = get_file_data( $style_path, array( 'Version' => 'Version' ), 'theme' );
			if ( is_array( $data ) && ! empty( $data['Version'] ) ) {
				$package_ver = (string) $data['Version'];
			}
		}

		$dest = trailingslashit( $theme_root ) . $stylesheet;
		self::delete_dir( $dest );
		wp_mkdir_p( $dest );
		$copy = copy_dir( $src, $dest );
		self::delete_dir( $working_dir );

		if ( is_wp_error( $copy ) ) {
			return $copy;
		}

		wp_clean_themes_cache( true );
		$after_ver   = (string) wp_get_theme( $stylesheet )->get( 'Version' );
		$after_ver_n = self::norm_ver( $after_ver );

		if ( '' !== $new_ver_n && '' !== $before_ver_n && version_compare( $new_ver_n, $before_ver_n, '>' ) ) {
			if ( '' === $after_ver_n || version_compare( $after_ver_n, $new_ver_n, '<' ) ) {
				$extra = '';
				if ( $package_ver ) {
					$extra = ' | Package: ' . $package_ver;
				}
				return new WP_Error(
					'pluxix_client_update_verify_failed',
					sprintf(
						/* translators: 1: old version, 2: expected new version, 3: detected version, 4: package version */
						esc_html__( 'Update completed but version did not change. Old: %1$s | Expected: %2$s | Detected: %3$s%4$s', 'pluxix-client' ),
						$before_ver ?: '-',
						$new_version ?: '-',
						$after_ver ?: '-',
						$extra
					)
				);
			}
		}

		return array( 'success' => true, 'version' => $after_ver, 'stylesheet' => $stylesheet, 'package_ver' => $package_ver );
	}
}
